
#include "WorkSpace.h"
#include "drawutil.h"
#include <GL/gl.h>


void WorkSpace::computeSelectionCentre() {
   if ( ! _selectedObjects.empty() ) {
      AlignedBox box;
      list< OIndex >::const_iterator it = _selectedObjects.begin();
      for ( ; it != _selectedObjects.end(); ++it ) {
         box.bound( _scene.getObject( *it )->getBoundingBox() );
      }
      _selectionCentre = box.getCentre();
   }
   else _selectionCentre = Point3(0,0,0);
}

void WorkSpace::draw( Scene::DrawMode drawMode ) const {
   glPushAttrib( GL_LIGHTING_BIT );
   glDisable( GL_LIGHTING );
   if ( _drawAxes ) {
      drawAxes( Point3(0,0,0), 1, true );
   }
   if ( _drawGroundPlane ) {
      // ... FIXME
   }
   if ( ! _selectedObjects.empty() ) {
      glColor3f( 0, 1, 1 );
      list< OIndex >::const_iterator it = _selectedObjects.begin();
      for ( ; it != _selectedObjects.end(); ++it ) {
         drawBox( _scene.getObject( *it )->getBoundingBox(), true, true );
      }
   }

   glColor3f( 1, 1, 1 );
   _scene.draw( drawMode );
   glPopAttrib();
}

void WorkSpace::xorSelection( list< OIndex >& l ) {
   list<OIndex>& L1 = _selectedObjects;
   list<OIndex>& L2 = l;

   L1.sort();
   L2.sort();
   L2.unique(); // eliminate possible duplicates in the input

   list<OIndex>::iterator i1 = L1.begin();
   list<OIndex>::const_iterator i2 = L2.begin();
   list<OIndex> L3;

   // for each element in L2, either remove the same element in L1 or
   // add the elment to L3
   while ( i2 != L2.end() ) {
      // try finding the element in L1
      while ( *i1 < *i2 && i1 != L1.end() ) ++i1;

      if ( i1 == L1.end() ) {
         // concatenate the rest of L2 onto L3
         do {
            L3.push_back( *i2 );
            ++i2;
         } while ( i2 != L2.end() );
      }
      else {
         if ( *i1 == *i2 ) {
            i1 = L1.erase( i1 );
         }
         else {
            L3.push_back( *i2 );
         }
         ++i2;
      }
   }

   // now concatenate L3 onto L1
   list<OIndex>::const_iterator i3 = L3.begin();
   while ( i3 != L3.end() ) {
      L1.push_back( *i3 );
      ++i3;
   }

   computeSelectionCentre();
}

void WorkSpace::insertTetrahedron() {
   Object * object = new Object();
   object->setToTetrahedron();
   _scene.add( object );
   _selectedObjects.clear();
   _selectedObjects.push_back( _scene.getNbObjects()-1 );
   computeSelectionCentre();
}

void WorkSpace::insertCube() {
   Object * object = new Object();
   object->setToCube();
   _scene.add( object );
   _selectedObjects.clear();
   _selectedObjects.push_back( _scene.getNbObjects()-1 );
   computeSelectionCentre();
}

void WorkSpace::insertOctahedron() {
   Object * object = new Object();
   object->setToOctahedron();
   _scene.add( object );
   _selectedObjects.clear();
   _selectedObjects.push_back( _scene.getNbObjects()-1 );
   computeSelectionCentre();
}

void WorkSpace::insertFlatRectangularMesh( int rows, int cols ) {
   Object * object = new Object();
   object->setToFlatRectangularMesh( rows, cols );
   _scene.add( object );
   _selectedObjects.clear();
   _selectedObjects.push_back( _scene.getNbObjects()-1 );
   computeSelectionCentre();
}

void WorkSpace::insertSphere( int lats, int lons ) {
   Object * object = new Object();
   object->setToSphere( lats, lons );
   _scene.add( object );
   _selectedObjects.clear();
   _selectedObjects.push_back( _scene.getNbObjects()-1 );
   computeSelectionCentre();
}

void WorkSpace::insertSeashell() {
   Object * object = new Object();
   object->setToSeashell();
   _scene.add( object );
   _selectedObjects.clear();
   _selectedObjects.push_back( _scene.getNbObjects()-1 );
   computeSelectionCentre();
}

void WorkSpace::translateSelection(
   const Vector3& v, bool onlyTransformLocalSpace
) {
   list< OIndex >::const_iterator it;
   for ( it = _selectedObjects.begin(); it != _selectedObjects.end(); ++it ) {
      _scene.getObject( *it )->translate( v, !onlyTransformLocalSpace, true );
   }
   if ( ! onlyTransformLocalSpace ) {
      _selectionCentre += v;
   }
}

void WorkSpace::rotateSelection(
   float angle_in_radians, const Vector3& axis,
   const Point3& rotationCentre, bool onlyTransformLocalSpace
) {
   Matrix m, m_inv;
   m.setToRotation( angle_in_radians, axis );
   m_inv.setToRotation( -angle_in_radians, axis );
   list< OIndex >::const_iterator it;
   for ( it = _selectedObjects.begin(); it != _selectedObjects.end(); ++it ) {
      _scene.getObject( *it )->rotate(
         m, m_inv, rotationCentre, !onlyTransformLocalSpace, true
      );
   }
   computeSelectionCentre();
}

void WorkSpace::scaleSelection(
   const Vector3& v, const Point3& scaleCentre, bool onlyTransformLocalSpace
) {
   // make sure the transformation isn't singular
   if ( fabs( v.get()[ v.indexOfLeastComponent() ] ) == 0 ) {
      // the scale can't be inverted, hence we can't handle it
      ASSERT( false );
      return;
   }

   Matrix m, m_inv;
   m.setToScale( v );
   m_inv.setToScale( Vector3( 1/v.x(), 1/v.y(), 1/v.z() ) );
   list< OIndex >::const_iterator it;
   for ( it = _selectedObjects.begin(); it != _selectedObjects.end(); ++it ) {
      _scene.getObject( *it )->scale(
         m, m_inv, scaleCentre, !onlyTransformLocalSpace, true
      );
   }
   computeSelectionCentre();
}

void WorkSpace::deleteSelection() {

   // As each object is deleted, the indices of later objects
   // will be shifted down one.  Hence, to make things easy,
   // just delete in reverse order.
   _selectedObjects.sort();
   _selectedObjects.reverse();

   list< OIndex >::const_iterator it;
   for ( it = _selectedObjects.begin(); it != _selectedObjects.end(); ++it ) {
      _scene.deleteObject( *it );
   }
   _selectedObjects.clear();
}

