
#include "SerialMouse.h"

#include <stdio.h>   // printf()
#include <termios.h> // tcgetattr(),tcsetattr(),cfsetispeed(),cfsetospeed()
#include <fcntl.h>   // open()
#include <unistd.h>  // read()
#include <errno.h>   // errno
#include <string.h>  // strerror()

#include <sys/ioctl.h> // I don't think this is necessary on Irix

#define BUFSIZE 256
#define MOUSE_SYNC_BYTE(b) ((b) & 0x40)


void printError( char * port, char * msg ) {
   printf("%s: %s (%s)\n", msg, strerror( errno ), port );
}


bool SerialMouse::init( char *port ) {

   struct termios term;

   if ( 0 > (_fd = open( port, O_RDONLY )) ) {
      printError( port, "couldn't open port" );
      return false;
   }
   if ( 0 > tcgetattr( _fd, &term ) ) {
      // couldn't retrieve terminal attributes
      printError( port, "couldn't tcgetattr()" );
      return false;
   }

   /* output flags */
   term.c_oflag = 0;

   /* input flags */
   term.c_iflag = IGNBRK | IGNPAR;

   /* control flags */
   term.c_cflag = CS7 | CREAD | CLOCAL;

   /* local flags */
   term.c_lflag = 0;

   /* control characters */
   term.c_cc[VMIN] = 0;
   term.c_cc[VTIME] = 1;

   /* baud rate */
   cfsetispeed(&term, B1200);
   cfsetospeed(&term, B1200);

   if ( 0 > tcsetattr( _fd, TCSANOW, &term ) ) {
      // couldn't change terminal attributes
      printError( port, "couldn't tcsetattr()" );
      return false;
   }
   return true;
}


bool SerialMouse::poll() {

   bool returnValue = false;
   size_t nbBytesPending = 0;
   int i;

   // check if any bytes are waiting in the input stream
   if ( 0 > ioctl( _fd, FIONREAD, &nbBytesPending ) ) {
      printError( "", "couldn't ioctl()" );
   }

#ifdef DEBUG_SERIAL_MOUSE
   //printf( "nbBytesPending == %d\n", (int)nbBytesPending );
#endif

   if ( nbBytesPending > 0 ) { 
      // read all pending input in one large gulp
      char buf[ BUFSIZE+3 ];
      int nbBytesRead = read( _fd, buf, BUFSIZE );

#ifdef DEBUG_SERIAL_MOUSE
      printf( "nbBytesRead == %d\n", nbBytesRead );

      for( i = 0; i < nbBytesRead; ++i ) {
         printf("%d%s ", (int)buf[i], MOUSE_SYNC_BYTE(buf[i]) ? "*" : "" );
      }
      printf("\n");
#endif

      for ( i = 0; i < nbBytesRead; ++i ) {
         if ( MOUSE_SYNC_BYTE(buf[i]) ) {
#ifdef DEBUG_SERIAL_MOUSE
            puts("got sync");
#endif
            _state = 1;
            _packet[0] = buf[i];
         }
         else {
            if ( _state == 0 ) {
               // We appear to be at an unknown position within a packet.
               // Normally, I would assume that we're here because we
               // somehow missed a sync byte and got lost ... in that case,
               // the easiest course of action would be to just discard
               // the data and wait for the next sync byte.

               // However, experimentation with my particular serial mouse
               // (a logitech serial mouse) shows that when the middle mouse
               // button changes state, a packet of 4 bytes is sent,
               // where the fourth byte is either 32 or 0 (for press or
               // release, respectively, of the MMB), even though this
               // is counter to the serial mouse protocol that this code
               // is based on. (The said protocol only supports 2-button mice.)

               // Hence I will hereby set the MMB state, and then wait for
               // the next sync byte.
               returnValue = true;
               //if ( buf[i] != 0 && buf[i] != 32 ) {
               // printf( "Unexpected serial mouse input %d\n", (int)buf[i] );
               //}
               middleBtn = buf[i] != 0;
               continue;
            }
            else {
               _packet[ _state ] = buf[i];
               _state++;

               if ( _state == 3 ) {
                  _state = 0;
                  // we have a full packet; we now interpret the packet

                  leftBtn = (_packet[0] & 0x20) >> 5;
                  rightBtn = (_packet[0] & 0x10) >> 4;

                  unsigned char dx = _packet[1] & 0x3F;
                  dx |= (_packet[0] << 6);

                  unsigned char dy = _packet[2] & 0x3F;
                  dy |= ((_packet[0] & 0x0C) << 4);

                  int deltaX;

                  if (dx & 0x80) {
                     dx = (dx ^ 0xFF);
                     dx += 1;
                     deltaX = -1 * dx;
                  }
                  else {
                     deltaX = dx;
                  }

                  int deltaY;
                  if (dy & 0x80) {
                     dy = (dy ^ 0xFF);
                     dy += 1;
                     deltaY = -1 * dy;
                  }
                  else {
                     deltaY = dy;
                  }

#ifdef DEBUG_SERIAL_MOUSE
                  printf("delta:(%d,%d)\n", deltaX, deltaY );
#endif
                  x += deltaX;
                  y += deltaY;

                  returnValue = true;
               }
            }
         }
      } // for

#ifdef DEBUG_SERIAL_MOUSE
      printf("(%d,%d,%d); pos==(%d,%d); %d, \"%d %d %d\"\n",
         leftBtn, middleBtn, rightBtn, x, y,
         _state, (int)(_packet[0]), (int)(_packet[1]), (int)(_packet[2])
      );
#endif
   }

   return returnValue;
}


SerialMouse::~SerialMouse() {
   close( _fd );
   _fd = -1;
}

