
#ifndef OBJECT_H
#define OBJECT_H


#include "mathutil.h"
#include <list>


typedef unsigned short FIndex; // Face Index
typedef unsigned short VIndex; // Vertex Index
#define MAX_INCIDENT_FACES_PER_VERTEX 8


class Triangle {
public:
   VIndex v[3]; // vertices, expressed as indices into an array of points
   Vector3 normal;
   bool normalIsDirty;
   Triangle( VIndex a, VIndex b, VIndex c ) : normalIsDirty( true ) {
      v[0] = a;  v[1] = b; v[2] = c;
   }
   void computeNormal( const vector< Point3 >& vertices ) {
      Vector3 v0( vertices[v[1]]-vertices[v[0]] );
      Vector3 v1( vertices[v[2]]-vertices[v[0]] );
      normal = ( v0 ^ v1 ).normalized();
      normalIsDirty = false;
   }
   // Returns true if there was an intersection
   bool intersects(
      const vector< Point3 >& vertices,
      const Ray& ray, float& t, bool cullBackfaces
   );
};


class VertexInfo {
public:
   FIndex incidentFaces[ MAX_INCIDENT_FACES_PER_VERTEX ];
   unsigned char numIncidentFaces;
   Vector3 normal;
   unsigned char bitFlags;
   VertexInfo() : numIncidentFaces( 0 ), bitFlags( 0 ) {
      setIncidentFaceListIsDirty( true );
      setNormalIsDirty( true );
   }
   bool incidentFaceListIsDirty() { return bitFlags & 1; }
   void setIncidentFaceListIsDirty( bool b ) {
      bitFlags = (bitFlags & 0xfe) | ( b ? 1 : 0 );
   }
   bool normalIsDirty() { return bitFlags & 2; }
   void setNormalIsDirty( bool b ) {
      bitFlags = (bitFlags & 0xfd) | ( b ? 2 : 0 );
   }

   void computeIncidentFaceList( VIndex me, const vector< Triangle >& faces ) {
      numIncidentFaces = 0;
      for ( FIndex f = 0; f < faces.size(); ++f ) {
         if ( faces[f].v[0]==me || faces[f].v[1]==me || faces[f].v[2]==me ) {
            ASSERT( numIncidentFaces < MAX_INCIDENT_FACES_PER_VERTEX );
            incidentFaces[ numIncidentFaces++ ] = f;
         }
      }
      setIncidentFaceListIsDirty( false );
   }
   void computeNormal(
      VIndex me,
      vector< Triangle >& faces, const vector< Point3 >& vertices
   ) {
      if ( incidentFaceListIsDirty() ) computeIncidentFaceList( me, faces );
      normal = Vector3(0,0,0);
      for ( int i = 0; i < numIncidentFaces; ++i ) {
         if ( faces[incidentFaces[i]].normalIsDirty )
            faces[incidentFaces[i]].computeNormal( vertices );
         normal += faces[incidentFaces[i]].normal;
      }
      normal = normal.normalized();
      setNormalIsDirty( false );
   }
};


// (Note: a "space" is defined by a point called the origin
// and an orthonormal basis called the frame.
// For 3-dimensional spaces, the frame is a set of 3 orthogonal unit vectors.
// If the 3-dimensional space is right-handed, then the cross product of
// the first two vectors in the frame yields the third vector.)
//
// Vertices are stored as points in Object Space O rather than
// as transformed points in World Space W.  This is done to
// prevent "numerical drift", i.e. points moving slightly closer
// or father together as they are iteratively transformed.
//
// We also distinguish a Local Space L.  L has its origin at
// the "pivot point", and initially L coincides with O.
// However, if the user moves the pivot point or rotates the
// local frame, L and O become distinct.  
//
// We are required to cache an axis-aligned bounding box in world space
// for the object.
// Note that changes in the object's orientation can change the
// bounding box's shape, and make the bounding box larger
// *or smaller* than it initially was.  Hence, a bounding box
// in O is not necessarily minimal, and therefore we will not
// try to compute the bounding box in W from a bounding box in O.
// On the other hand, we will not store the bounding box in W
// and apply transformations directly to it, because
// (as with the vertices) iterated transformations could
// create "numerical drift".
// Instead, we introduce an Intermediate Space I.  I is the
// result of applying rotation (and scale) transformations to O,
// but not translations.  The frame of I coincides with the
// frame of W.  We store a bounding box in I, and recompute it
// from the object's vertices whenever the object is rotated.
// We can easily compute the bounding box in W from the one
// in I simply by applying a translation.
// (Note: the origin of I does *not* coincide with the origin
// of O.  This would be the case if all rotations were centred
// on O's origin, but we centre rotations on L's origin
// (i.e. the "piovt point").)

class Object {
private:
   vector< Point3 > _vertices;  // in object space
   vector< Triangle > _faces;
   vector< VertexInfo > _vertexInfo;

   // To get from object space to world space,
   // we first apply the "o2i" transform
   // (to get from object space to intermediate space),
   // and then apply the "i2w" translation
   // (to get from intermediate space to world space).
   //
   // Note that the "o2i" transform contains the cumulative rotation
   // and scale transforms that the object has undergone.
   // However, each rotation transformation is centered on
   // the "pivot point" (i.e. the local origin), hence the point
   // (0,0,0) is not generally invariant under the "o2i" transform.
   // (Actually, in general, the "o2i" transform has no fixed points.)
   // Therefore, the object origin in world space must be computed as
   //    _O2ITransform * Point3(0,0,0) + _I2WTranslation
   // and not as
   //    Point3(0,0,0) + _I2WTranslation
   // (which would actually yield the intermediate origin in world space).
   //
   // FIXME: it's bad to store cumulative transformations as a matrix.
   // It can suffer from drift.  A better representation would be
   // translation, rotation, scale, and shear factors, stored not as
   // matrices but as something from which matrices can be generated.
   //
   Matrix _O2ITransform, _O2ITransform_inverse;
   Vector3 _I2WTranslation;

   // The local origin (or "pivot point"), expressed
   // in terms of intermediate space.
   //
   // The local origin in world space can be computed as
   //    _LOrigin_I + _I2WTranslation
   //
   Point3 _LOrigin_I;

   // The local frame, expressed in terms of world space.
   // FIXME: this could suffer from drift.  We should store
   // the frame's orientation as a quaternion.
   Vector3 _LFrame_W[3];

   // The bounding box, in intermediate space.
   AlignedBox _intermediateBoundingBox;
   bool _intermediateBoundingBoxDirty;

   // The bounding box, in world space.
   AlignedBox _worldBoundingBox;
   bool _worldBoundingBoxDirty;
public:
   enum DistanceMetric {
      // Euclidean distance
      SPHERICAL_DISTANCE,

      // distance from point to axis of cylinder
      // (or, equivalently, distance after projecting
      // point onto a plane)
      CYLINDRICAL_DISTANCE
   };
private:
   void visitNeighbours(
      VIndex vertexIndex,
      const Point3& centre,
      float radiusSquared,
      DistanceMetric distanceMetric,
      const Vector3& normal,
      vector< bool >& visitedVertices,
      list< VIndex >& neighbouringVertices,
      list< float >& vertexDistances,
      vector< bool >& visitedFaces,
      list< FIndex >& neighbouringFaces
   );
public:
   Object() :
      _intermediateBoundingBoxDirty( true ), _worldBoundingBoxDirty( true )
   {
      _LFrame_W[0] = Vector3( 1, 0, 0 );
      _LFrame_W[1] = Vector3( 0, 1, 0 );
      _LFrame_W[2] = Vector3( 0, 0, 1 );
   }

   Point3 getLocalOrigin() const { return _LOrigin_I+_I2WTranslation; }
   void getLocalFrame( Vector3& i, Vector3& j, Vector3& k ) const {
      i = _LFrame_W[0];  j = _LFrame_W[1];  k = _LFrame_W[2];
   }

   const AlignedBox& getBoundingBox();

   const VIndex getNbVertices() const { return _vertices.size(); }
   Point3 getVertex( VIndex i ) const {
      return _O2ITransform * _vertices[i] + _I2WTranslation;
   }
   void setVertex( VIndex vertexIndex, const Point3& );

   void setToTetrahedron();
   void setToCube();
   void setToOctahedron();
   void setToFlatRectangularMesh( int rows, int cols );
   void setToSphere( int lats, int lons );
   void setToSeashell();

   void draw(
      bool drawGeometry,
      bool outputFaceNormals,
      bool outputVertexNormals,
      bool drawLocalFrame = false,
      float lengthOfLocalFrame = 1
   );

   void translate(
      const Vector3& v,  // in world space
      bool transformGeometry,
      bool transformLocalSpace
   );

   void rotate(
      const Matrix& m, // the corresponding rotation matrix, centred on (0,0,0)
      const Matrix& m_inverse,
      const Point3& rotationCentre, // in world space
      bool transformGeometry,
      bool transformLocalSpace
   );

   void scale(
      const Matrix& m, const Matrix& m_inverse, // centred on (0,0,0)
      const Point3& scaleCentre, // in world space
      bool transformGeometry,
      bool transformLocalSpace
   );

   // FIXME: any future "translate vertex" method should soil the normals

   void reset();

   // If there are intersections, returns a list of faces and a
   // corresponding list of ray parameter values.  Both lists
   // are ordered from nearest to farthest intersection.
   // If there are no intersections, returns false.
   bool intersectAll(
      const Ray& ray,
      list< FIndex >& face_list, list< float >& t_list,
      bool cullBackfaces
   );

   // If there are intersections, returns the nearest intersected face
   // and the corresponding ray parameter value.
   // If there are no intersections, returns false.
   bool intersectNearest(
      const Ray& ray, FIndex & face, float & t, bool cullBackfaces
   ) {
      list< FIndex > face_list;
      list< float > t_list;
      bool b = intersectAll( ray, face_list, t_list, cullBackfaces );
      if ( b ) { face = face_list.front(); t = t_list.front(); }
      return b;
   }

   Vector3 getFaceNormal( FIndex faceIndex ) {
      Triangle * face = &_faces[ faceIndex ];
      if ( face->normalIsDirty ) face->computeNormal( _vertices );

      // To transform *points* from object space to intermediate space,
      // we use the "o2i" transform.
      // Hence, to transform *normals* from object space to intermediate space,
      // we use the transpose of the inverse of the "o2i" transform.
      // Note, however, that w has to be reset to 1, and the normal has
      // to be renormalized.
      // Note also that, since intermediate space and world space only
      // differ by a translation, nothing further is required to transform
      // the normal to world space.
      Matrix M = _O2ITransform_inverse;
      M.transpose();
      Vector3 n = M * face->normal;
      n.w() = 1;
      return n.normalized();
   }
   Vector3 getVertexNormal( VIndex vertexIndex ) {
      VertexInfo * vi = &_vertexInfo[ vertexIndex ];
      if ( vi->normalIsDirty() ) vi->computeNormal(
         vertexIndex, _faces, _vertices
      );
      Matrix M = _O2ITransform_inverse;
      M.transpose();
      Vector3 n = M * vi->normal;
      n.w() = 1;
      return n.normalized();
   }

   // Returns the normal (computed by interpolating the vertex normals)
   // at the given point on the given face.
   Vector3 getNormal( FIndex, const Point3& );

   // Returns an unsorted list of the vertices
   // (and a corresponding list of distances)
   // that fall within the given radius of the given centre point.
   // An unsorted list of faces incident on the vertices is also returned.
   //
   // If a cylindrical distance metric is requested, the vertices
   // are projected onto a plane perpendicular to the given normal,
   // and tested to see if they fall within a disc of the given radius.
   //
   // To speed up the search, the client provides a "starting face".
   // Typically, the centre point lies on the starting face.
   // This subroutine searches outward from the starting face, and
   // stops when it reaches vertices that fall outside the given radius.
   // Hence, the subroutine may not find all vertices that lie within
   // the radius.  It will only find vertices if they are "connected"
   // to the starting face via a path of vertices that are all
   // within the radius.
   void computeNeighbourhood(
      const Point3& centre,
      float radius,
      DistanceMetric distanceMetric,
      const Vector3& normal, // used for cylindrical distance metric
      FIndex startingFace,
      list< VIndex >& neighbouringVertices,
      list< float >& vertexDistances,
      list< FIndex >& neighbouringFaces
   );
};


#endif /* OBJECT_H */

