
#include "Object.h"
#include "drawutil.h"
#include <GL/gl.h>


// Projects the 4 given points onto one of the axis-aligned planes,
// and returns the coordinates as (s,t).
// dimension is one of 0,1,2 for the yz, zx, or xy plane, respectively.
// If flipHandedness is true, s and t are swapped.
inline void project(
   const Point3& p0, const Point3& p1, const Point3& p2, const Point3& p3,
   float & s0, float & t0,
   float & s1, float & t1,
   float & s2, float & t2,
   float & s3, float & t3,
   short dimension,
   bool flipHandedness
) {
   switch ( dimension ) {
      case 0 : // project onto the yz plane
         if ( flipHandedness ) {
            s0 = p0.z();
            t0 = p0.y();
            s1 = p1.z();
            t1 = p1.y();
            s2 = p2.z();
            t2 = p2.y();
            s3 = p3.z();
            t3 = p3.y();
         }
         else {
            s0 = p0.y();
            t0 = p0.z();
            s1 = p1.y();
            t1 = p1.z();
            s2 = p2.y();
            t2 = p2.z();
            s3 = p3.y();
            t3 = p3.z();
         }
         break;
      case 1 : // project onto the zx plane
         if ( flipHandedness ) {
            s0 = p0.x();
            t0 = p0.z();
            s1 = p1.x();
            t1 = p1.z();
            s2 = p2.x();
            t2 = p2.z();
            s3 = p3.x();
            t3 = p3.z();
         }
         else {
            s0 = p0.z();
            t0 = p0.x();
            s1 = p1.z();
            t1 = p1.x();
            s2 = p2.z();
            t2 = p2.x();
            s3 = p3.z();
            t3 = p3.x();
         }
         break;
      case 2 : // project onto the xy plane
         if ( flipHandedness ) {
            s0 = p0.y();
            t0 = p0.x();
            s1 = p1.y();
            t1 = p1.x();
            s2 = p2.y();
            t2 = p2.x();
            s3 = p3.y();
            t3 = p3.x();
         }
         else {
            s0 = p0.x();
            t0 = p0.y();
            s1 = p1.x();
            t1 = p1.y();
            s2 = p2.x();
            t2 = p2.y();
            s3 = p3.x();
            t3 = p3.y();
         }
         break;
      default : ASSERT( false );
   }
}


bool Triangle::intersects(
   const vector< Point3 >& vertices,
   const Ray& ray, float& t, bool cullBackfaces
) {
   if ( normalIsDirty ) computeNormal( vertices );
   float dot = normal * ray.direction;
   if ( cullBackfaces && dot > 0 ) {
      return false;
   }
   if ( dot == 0 ) {
      return false;
   }

   // See Foley and van Dam, pg 1101
   t = ( vertices[v[0]] - ray.origin ) * normal / dot;

   if ( t < 0 ) {
      return false;
   }

   Point3 intersection = ray.point( t );

   // Figure out which of the 3 axis-aligned planes is most
   // parallel to the triangle.
   short dimension = normal.indexOfGreatestComponent();

   // Now, project the triangle's vertices and the intersection
   // point down onto an s,t plane that coincides with the
   // axis-aligned plane most parallel to the triangle.
   // Be careful to preserve handedness of the space.
   float s0, t0, s1, t1, s2, t2;  // the triangle's vertices
   float si, ti;  // the intersection point
   project(
      vertices[v[0]], vertices[v[1]], vertices[v[2]], intersection,
      s0, t0, s1, t1, s2, t2, si, ti,
      dimension, normal.get()[ dimension ] < 0
   );

   // Now, do three "left-hand-turn" tests
   // to see if the intersection point is inside the triangle.
   // (Side note: Given two vectors a and b
   // that lie on the xy plane of a right-handed 3-space,
   // b is a left-hand-turn with respect to a
   // if the z component of (a x b) is positive.
   // Equivalently, given a directed line segment ab and a point p
   // that lie on the xy plane of a right-handed 3-space,
   // p is on the left-hand-side of ab
   // if the z component of (b-a)x(p-b) is positive.)

   // FIXME: maybe this should be packaged up into a
   // Vector2 class or something.
   #define LEFT_HAND_TEST(ax,ay,bx,by,px,py) ( \
      ((bx)-(ax))*((py)-(by)) - ((by)-(ay))*((px)-(bx)) > 0 \
   )
   return (
      LEFT_HAND_TEST(s0,t0,s1,t1,si,ti) && 
      LEFT_HAND_TEST(s1,t1,s2,t2,si,ti) &&
      LEFT_HAND_TEST(s2,t2,s0,t0,si,ti)
   );
   #undef LEFT_HAND_TEST
}


// FIXME: since this routine is recursive, we should try to make
// calls to it efficient.  Unfortunately, a lot of arguments have
// to be passed in for each call.  We could increase efficiency
// by storing the arguments that don't change (i.e. centre,
// radiusSquared, distanceMetric, normal, and the vector<>s and
// list<>s that are passed by reference) in a struct and passing
// one pointer for the whole struct.
void Object::visitNeighbours(
   VIndex vertexIndex,
   const Point3& centre,
   float radiusSquared,
   DistanceMetric distanceMetric, // FIXME: we ignore this parameter
   const Vector3& normal,  // FIXME: we ignore this parameter
   vector< bool >& visitedVertices,
   list< VIndex >& neighbouringVertices,
   list< float >& vertexDistances,
   vector< bool >& visitedFaces,
   list< FIndex >& neighbouringFaces
) {
   ASSERT( ! visitedVertices[ vertexIndex ] );
   visitedVertices[ vertexIndex ] = true;

   // FIXME: you've only implemented the spherical distance metric.
   float d = (_vertices[ vertexIndex ] - centre).lengthSquared();
   if ( d > radiusSquared ) return;
   d = sqrt( d );
   neighbouringVertices.push_back( vertexIndex );
   vertexDistances.push_back( d );

   VertexInfo * vi = &_vertexInfo[ vertexIndex ];
   if ( vi->incidentFaceListIsDirty() )
      vi->computeIncidentFaceList( vertexIndex, _faces );
   for ( int i = 0; i < vi->numIncidentFaces; ++i ) {
      FIndex faceIndex = vi->incidentFaces[i];
      if ( visitedFaces[ faceIndex ] ) continue;
      visitedFaces[ faceIndex ] = true;
      neighbouringFaces.push_back( faceIndex );
      
      if ( ! visitedVertices[ _faces[ faceIndex ].v[0] ] )
         visitNeighbours(
            _faces[ faceIndex ].v[0],
            centre, radiusSquared, distanceMetric, normal,
            visitedVertices, neighbouringVertices, vertexDistances,
            visitedFaces, neighbouringFaces
         );
      if ( ! visitedVertices[ _faces[ faceIndex ].v[1] ] )
         visitNeighbours(
            _faces[ faceIndex ].v[1],
            centre, radiusSquared, distanceMetric, normal,
            visitedVertices, neighbouringVertices, vertexDistances,
            visitedFaces, neighbouringFaces
         );
      if ( ! visitedVertices[ _faces[ faceIndex ].v[2] ] )
         visitNeighbours(
            _faces[ faceIndex ].v[2],
            centre, radiusSquared, distanceMetric, normal,
            visitedVertices, neighbouringVertices, vertexDistances,
            visitedFaces, neighbouringFaces
         );
   }
}

const AlignedBox& Object::getBoundingBox() {
   if ( _worldBoundingBoxDirty ) {
      if ( _intermediateBoundingBoxDirty ) {
         _intermediateBoundingBox.clear();
         _intermediateBoundingBox.bound( _vertices, _O2ITransform );
         _intermediateBoundingBoxDirty = false;
      }
      _worldBoundingBox = AlignedBox(
         _intermediateBoundingBox.getMin() + _I2WTranslation,
         _intermediateBoundingBox.getMax() + _I2WTranslation
      );
      _worldBoundingBoxDirty = false;
   }
   else ASSERT( ! _intermediateBoundingBoxDirty );

   return _worldBoundingBox;
}

void Object::setVertex( VIndex vertexIndex, const Point3& p ) {
   Point3 intermediatePoint = p - _I2WTranslation;
   _vertices[ vertexIndex ] = _O2ITransform_inverse * intermediatePoint;
   VertexInfo * vi = &_vertexInfo[ vertexIndex ];
   vi->setNormalIsDirty( true );
   if ( vi->incidentFaceListIsDirty() )
      vi->computeIncidentFaceList( vertexIndex, _faces );
   for ( int i = 0; i < vi->numIncidentFaces; ++i ) {
      Triangle * tri = &_faces[vi->incidentFaces[i]];
      tri->normalIsDirty = true;
      _vertexInfo[ tri->v[0] ].setNormalIsDirty( true );
      _vertexInfo[ tri->v[1] ].setNormalIsDirty( true );
      _vertexInfo[ tri->v[2] ].setNormalIsDirty( true );
   }
   if (
      ! _intermediateBoundingBoxDirty
      && ! _intermediateBoundingBox.contains( intermediatePoint )
   ) {
      // The vertex has moved outside the bounding box, and is
      // the only one to have done so, so we can easily adjust
      // the bounding box without having to mark it dirty.
      _intermediateBoundingBox.bound( intermediatePoint );
   }
   else _intermediateBoundingBoxDirty = true;

   _worldBoundingBoxDirty = true;
}

void Object::setToTetrahedron() {
   _vertices.clear();
   _faces.clear();

   _vertices.push_back( Point3( -1, -1, -1 ) );
   _vertices.push_back( Point3(  1,  1, -1 ) );
   _vertices.push_back( Point3( -1,  1,  1 ) );
   _vertices.push_back( Point3(  1, -1,  1 ) );

   _faces.push_back( Triangle( 0, 2, 1 ) );
   _faces.push_back( Triangle( 0, 1, 3 ) );
   _faces.push_back( Triangle( 0, 3, 2 ) );
   _faces.push_back( Triangle( 1, 2, 3 ) );

   _vertexInfo.resize( _vertices.size() );

   _intermediateBoundingBox
      = AlignedBox( Point3( -1, -1, -1 ), Point3( 1, 1, 1 ) );
   _intermediateBoundingBoxDirty = false;
   _worldBoundingBoxDirty = true;
}

void Object::setToCube() {

   _vertices.clear();
   _faces.clear();

   _vertices.push_back( Point3( -1, -1, -1 ) );
   _vertices.push_back( Point3(  1, -1, -1 ) );
   _vertices.push_back( Point3( -1,  1, -1 ) );
   _vertices.push_back( Point3(  1,  1, -1 ) );
   _vertices.push_back( Point3( -1, -1,  1 ) );
   _vertices.push_back( Point3(  1, -1,  1 ) );
   _vertices.push_back( Point3( -1,  1,  1 ) );
   _vertices.push_back( Point3(  1,  1,  1 ) );

   _faces.push_back( Triangle( 0, 2, 3 ) );
   _faces.push_back( Triangle( 0, 3, 1 ) );
   _faces.push_back( Triangle( 0, 1, 5 ) );
   _faces.push_back( Triangle( 0, 5, 4 ) );
   _faces.push_back( Triangle( 0, 4, 6 ) );
   _faces.push_back( Triangle( 0, 6, 2 ) );
   _faces.push_back( Triangle( 1, 3, 7 ) );
   _faces.push_back( Triangle( 1, 7, 5 ) );
   _faces.push_back( Triangle( 2, 6, 7 ) );
   _faces.push_back( Triangle( 2, 7, 3 ) );
   _faces.push_back( Triangle( 4, 5, 7 ) );
   _faces.push_back( Triangle( 4, 7, 6 ) );

   _vertexInfo.resize( _vertices.size() );

   _intermediateBoundingBox
      = AlignedBox( Point3( -1, -1, -1 ), Point3( 1, 1, 1 ) );
   _intermediateBoundingBoxDirty = false;
   _worldBoundingBoxDirty = true;
}

void Object::setToOctahedron() {

   _vertices.clear();
   _faces.clear();

   _vertices.push_back( Point3( -1,  0,  0 ) );
   _vertices.push_back( Point3(  1,  0,  0 ) );
   _vertices.push_back( Point3(  0, -1,  0 ) );
   _vertices.push_back( Point3(  0,  1,  0 ) );
   _vertices.push_back( Point3(  0,  0, -1 ) );
   _vertices.push_back( Point3(  0,  0,  1 ) );

   _faces.push_back( Triangle( 0, 4, 2 ) );
   _faces.push_back( Triangle( 0, 2, 5 ) );
   _faces.push_back( Triangle( 0, 3, 4 ) );
   _faces.push_back( Triangle( 0, 5, 3 ) );
   _faces.push_back( Triangle( 1, 2, 4 ) );
   _faces.push_back( Triangle( 1, 5, 2 ) );
   _faces.push_back( Triangle( 1, 4, 3 ) );
   _faces.push_back( Triangle( 1, 3, 5 ) );

   _vertexInfo.resize( _vertices.size() );

   _intermediateBoundingBox
      = AlignedBox( Point3( -1, -1, -1 ), Point3( 1, 1, 1 ) );
   _intermediateBoundingBoxDirty = false;
   _worldBoundingBoxDirty = true;
}

void Object::setToFlatRectangularMesh( int rows, int cols ) {

   _vertices.clear();
   _faces.clear();

   int row, col;

   for ( row = 0; row <= rows; ++row ) {
      for ( col = 0; col <= cols; ++col ) {
         _vertices.push_back( Point3( row-0.5*rows, 0, col-0.5*cols ) );
      }
   }

   int ul, ur, ll, lr;  // indices of upper/lower left/right corners
   ul = 0; ur = 1;
   ll = cols+1;  lr = cols+2;
   for ( row = 0; row < rows; ++row ) {
      for ( col = 0; col < cols; ++col ) {
         _faces.push_back( Triangle( ur, ll, ul ) );
         _faces.push_back( Triangle( ll, ur, lr ) );
         ++ul; ++ur;
         ++ll; ++lr;
      }
      ++ul; ++ur;
      ++ll; ++lr;
   }

   _vertexInfo.resize( _vertices.size() );

   _intermediateBoundingBox = AlignedBox(
      Point3( -0.5*rows, 0, -0.5*cols ), Point3( 0.5*rows, 0, 0.5*cols )
   );
   _intermediateBoundingBoxDirty = false;
   _worldBoundingBoxDirty = true;
}

void Object::setToSphere( int lats, int lons ) {

   static const float r = 5;

   _vertices.clear();
   _faces.clear();

   int lat, lon;

   for ( lat = 0; lat < lats; ++lat ) {
      float latitude = (2*M_PI*lat)/lats;
      float x = r * cos( latitude );
      float z = r * sin( latitude );
      for ( lon = 1; lon < lons; ++lon ) {
         float longitude = M_PI*(lon/(float)lons - 0.5);
         float y = r * sin( longitude );
         float k = cos( longitude );
         _vertices.push_back( Point3( k*x, y, k*z ) );
      }
   }

   int ul, ur, ll, lr;  // indices of upper/lower left/right corners
   ll = 0; ul = 1;
   lr = lons-1; ur = lons;
   for ( lat = 0; lat < lats; ++lat ) {
      for ( lon = 1; lon < lons-1; ++lon ) {
         _faces.push_back( Triangle( ur, ll, ul ) );
         _faces.push_back( Triangle( ll, ur, lr ) );
         ++ul; ++ur;
         ++ll; ++lr;
      }
      ++ul; ++ur;
      ++ll; ++lr;
      if ( (unsigned)lr >= _vertices.size() ) {
         lr = 0; ur = 1;
      }
   }

#if 0 // although this works, it makes too many triangles incident on each pole
   unsigned int northPole = _vertices.size();
   _vertices.push_back( Point3( 0, r, 0 ) );  // north pole
   _vertices.push_back( Point3( 0, -r, 0 ) ); // south pole

   ul = lons-2; ur = 2*lons-3;
   ll = 0; lr = lons-1;
   for ( lat = 0; lat < lats; ++lat ) {
      _faces.push_back( Triangle( ur, ul, northPole ) );
      _faces.push_back( Triangle( ll, lr, northPole+1 ) );
      ul = ur;
      ll = lr;
      ur += lons-1;
      lr += lons-1;
      if ( lat == lats-2 ) {
         lr = 0;  ur = lons-2;
      }
   }
#endif

   _vertexInfo.resize( _vertices.size() );

   _intermediateBoundingBox = AlignedBox(
      Point3( -r, -r, -r ), Point3( r, r, r )
   );
   _intermediateBoundingBoxDirty = false;
   _worldBoundingBoxDirty = true;
}

void Object::setToSeashell() {

   _vertices.clear();
   _faces.clear();

#if 0
   int nbRotations = 4;
   int nbLat = 10;
   int nbLongPerRotation = 10;
   int long, lat;
   float R0 = 3, r0 = 1;
   float R, r;
   for ( long = 0; long < nbLongPerRotation * nbRotations; ++long ) {
      float phi = 2*M_PI*long/(float)nbLongPerRotation;
      float k = 
      for ( lat = 0; lat < nbLat; ++lat ) {
         float theta = 2*M_PI*lat/(float)nbLat;
         //... FIXME
      }
   }
#endif

   _intermediateBoundingBoxDirty = true;
   _worldBoundingBoxDirty = true;
}

void Object::draw(
   bool drawGeometry, bool outputFaceNormals, bool outputVertexNormals,
   bool drawLocalFrame, float lengthOfLocalFrame
) {
   ASSERT( drawGeometry || drawLocalFrame );
   ASSERT( !outputFaceNormals || !outputVertexNormals );
   glPushMatrix();
      glTranslatef(
         _I2WTranslation.x(), _I2WTranslation.y(), _I2WTranslation.z()
      );
      if ( drawLocalFrame ) {
         glPushAttrib( GL_LIGHTING_BIT );
         glDisable( GL_LIGHTING );
         drawFrame(
            _LOrigin_I, _LFrame_W[0], _LFrame_W[1], _LFrame_W[2],
            lengthOfLocalFrame, false, false
         );
         glPopAttrib();
      }
      if ( drawGeometry ) {
         glMultMatrixf( _O2ITransform.get() );
         glBegin( GL_TRIANGLES );
            vector< Triangle >::iterator it = _faces.begin();
            if ( outputFaceNormals ) {
               for ( ; it != _faces.end(); ++it ) {
                  if ( (*it).normalIsDirty ) (*it).computeNormal( _vertices );
                  glNormal3fv( (*it).normal.get() );

                  glVertex3fv( _vertices[ (*it).v[0] ].get() );
                  glVertex3fv( _vertices[ (*it).v[1] ].get() );
                  glVertex3fv( _vertices[ (*it).v[2] ].get() );
               }
            }
            else if ( outputVertexNormals ) {
               VertexInfo * vi = 0;
               for ( VIndex i = 0; i < _vertices.size(); ++i ) {
                  vi = &_vertexInfo[ i ];
                  if ( vi->normalIsDirty() )
                     vi->computeNormal( i, _faces, _vertices );
               }
               for ( ; it != _faces.end(); ++it ) {
                  vi = &_vertexInfo[ (*it).v[0] ];
                  ASSERT( ! vi->normalIsDirty() );
                  glNormal3fv( vi->normal.get() );
                  glVertex3fv( _vertices[ (*it).v[0] ].get() );

                  vi = &_vertexInfo[ (*it).v[1] ];
                  ASSERT( ! vi->normalIsDirty() );
                  glNormal3fv( vi->normal.get() );
                  glVertex3fv( _vertices[ (*it).v[1] ].get() );

                  vi = &_vertexInfo[ (*it).v[2] ];
                  ASSERT( ! vi->normalIsDirty() );
                  glNormal3fv( vi->normal.get() );
                  glVertex3fv( _vertices[ (*it).v[2] ].get() );
               }
            }
            else {
               for ( ; it != _faces.end(); ++it ) {
                  glVertex3fv( _vertices[ (*it).v[0] ].get() );
                  glVertex3fv( _vertices[ (*it).v[1] ].get() );
                  glVertex3fv( _vertices[ (*it).v[2] ].get() );
               }
            }
         glEnd();
      }
   glPopMatrix();
}

void Object::translate(
   const Vector3& v, 
   bool transformGeometry,
   bool transformLocalSpace 
) {
   ASSERT( transformGeometry || transformLocalSpace );
   if ( transformGeometry ) {
      _I2WTranslation += v;
      _worldBoundingBoxDirty = true;
      if ( ! transformLocalSpace ) {
         _LOrigin_I -= v;
      }
   }
   else if ( transformLocalSpace ) {
      _LOrigin_I += v;
   }
}

void Object::rotate( 
   const Matrix& m, // the corresponding rotation matrix, centred on (0,0,0)
   const Matrix& m_inverse,
   const Point3& rotationCentre, // in world space
   bool transformGeometry,
   bool transformLocalSpace
) {
   ASSERT( transformGeometry || transformLocalSpace );
   Vector3 v = Vector3( rotationCentre - _I2WTranslation );
   Matrix T, T_inverse;
   T.setToTranslation( -v );
   T_inverse.setToTranslation( v );
   Matrix M;
   M = T_inverse * m * T;

   if ( transformLocalSpace ) {
      _LOrigin_I = M * _LOrigin_I;
      _LFrame_W[0] = m * _LFrame_W[0];
      _LFrame_W[1] = m * _LFrame_W[1];
      _LFrame_W[2] = m * _LFrame_W[2];
   }

   if ( transformGeometry ) {
      Matrix M_inverse;
      M_inverse = T_inverse * m_inverse * T;
      _O2ITransform = M * _O2ITransform;
      _O2ITransform_inverse = _O2ITransform_inverse * M_inverse;
      _worldBoundingBoxDirty = _intermediateBoundingBoxDirty = true;
   }
}

void Object::scale( 
   const Matrix& m, const Matrix& m_inverse,
   const Point3& scaleCentre,
   bool transformGeometry,
   bool transformLocalSpace
) {
   ASSERT( transformGeometry || transformLocalSpace );
   Vector3 v = Vector3( scaleCentre - _I2WTranslation );
   Matrix T, T_inverse;
   T.setToTranslation( -v );
   T_inverse.setToTranslation( v );
   Matrix M;
   M = T_inverse * m * T;

   if ( transformLocalSpace ) {
      _LOrigin_I = M * _LOrigin_I;
      // do *not* scale the local frame
   }

   if ( transformGeometry ) {
      Matrix M_inverse;
      M_inverse = T_inverse * m_inverse * T;
      _O2ITransform = M * _O2ITransform;
      _O2ITransform_inverse = _O2ITransform_inverse * M_inverse;
// FIXME: rather than marking the intermediate bounding box
// as dirty, is there any way we could simply scale a cached
// bounding box ?  Would this require us to store the "o2i"
// transform as translation/rotation/scale/shear components ?
// Is it possible to factor the transform in such a way ?
      _worldBoundingBoxDirty = _intermediateBoundingBoxDirty = true;
   }
}

void Object::reset() {
   _O2ITransform.setToIdentity();
   _O2ITransform_inverse.setToIdentity();
   _I2WTranslation = Vector3(0,0,0);

   _LOrigin_I = Point3( 0, 0, 0 );
   _LFrame_W[0] = Vector3( 1, 0, 0 );
   _LFrame_W[1] = Vector3( 0, 1, 0 );
   _LFrame_W[2] = Vector3( 0, 0, 1 );

   _worldBoundingBoxDirty = _intermediateBoundingBoxDirty = true;
}

bool Object::intersectAll(
   const Ray& ray,
   list< FIndex >& face_list, list< float >& t_list,
   bool cullBackfaces
) {
   ASSERT( face_list.empty() && t_list.empty() );

   // Transform the ray to object space.
   Ray r;
   r.origin = _O2ITransform_inverse * ( ray.origin - _I2WTranslation );
   // Note that the ray's direction should not be transformed as a normal
   // vector.  Rather, we should transform a point on the ray, and subtract
   // the transformed origin to get back the transformed direction.
   Point3 p = ray.origin + ray.direction;
   p = _O2ITransform_inverse * ( p - _I2WTranslation );
   r.direction = p - r.origin;
   float K = r.direction.length();
   r.direction = r.direction.normalized();

   // Loop over all the faces
   float t;
   for ( FIndex i = 0; i < _faces.size(); ++i ) {
      if ( _faces[i].intersects( _vertices, r, t, cullBackfaces ) ) {

         // Convert the parameter value in object space
         // to a parameter value in world space.
         // FIXME: should this be necessary ?  If we hadn't normalized
         // the ray's direction in object space, we could avoid having
         // to do any conversion of parameter values.  Is it necessary
         // to always have normalized ray directions ?
         t /= K;

         // Add i and t to the lists, keeping the lists ordered
         // from nearest to farthest intersection

         if ( t_list.empty() || t_list.back() <= t ) {
            face_list.push_back( i );
            t_list.push_back( t );
         }
         else {
            list< float >::iterator it = t_list.begin();
            list< FIndex >::iterator it2 = face_list.begin();
            while ( *it < t ) {
               ++it;
               ++it2;
               ASSERT( it != t_list.end() && it2 != face_list.end() );
            }
            t_list.insert( it, t );
            face_list.insert( it2, i );
         }
      }
   }

   ASSERT( face_list.size() == t_list.size() );

   return ! face_list.empty();
}

Vector3 Object::getNormal( FIndex faceIndex, const Point3& p ) {

   // Transform the point to object space
   Point3 P = _O2ITransform_inverse * ( p - _I2WTranslation );

   // Get the corners of the triangle
   Triangle * tri = &_faces[ faceIndex ];
   const Point3& p0 = _vertices[tri->v[0]];
   const Point3& p1 = _vertices[tri->v[1]];
   const Point3& p2 = _vertices[tri->v[2]];

   // Project the 4 points onto the axis-aligned plane
   // most parallel to the triangle.
   if ( tri->normalIsDirty ) tri->computeNormal( _vertices );
   short dimension = tri->normal.indexOfGreatestComponent();
   float s0, t0, s1, t1, s2, t2;  // the triangle's vertices
   float s, t;
   project(
      p0, p1, p2, P,
      s0, t0, s1, t1, s2, t2, s, t,
      dimension, tri->normal.get()[ dimension ] < 0
   );

   // Compute barycentric coordinates.
   float b  =  (s1 - s0)*(t2-t0) - (s2-s0)*(t1-t0);
   float b0 = ((s1 - s )*(t2-t ) - (s2-s )*(t1-t )) / b;
   float b1 = ((s2 - s )*(t0-t ) - (s0-s )*(t2-t )) / b;
   float b2 = ((s0 - s )*(t1-t ) - (s1-s )*(t0-t )) / b;

   // Make sure the vertex normals are up-to-date.
   VertexInfo * vi0 = &_vertexInfo[tri->v[0]];
   if ( vi0->normalIsDirty() ) vi0->computeNormal(
      tri->v[0], _faces, _vertices
   );
   VertexInfo * vi1 = &_vertexInfo[tri->v[1]];
   if ( vi1->normalIsDirty() ) vi1->computeNormal(
      tri->v[1], _faces, _vertices
   );
   VertexInfo * vi2 = &_vertexInfo[tri->v[2]];
   if ( vi2->normalIsDirty() ) vi2->computeNormal(
      tri->v[2], _faces, _vertices
   );

   // Perform weighted sum.
   Vector3 n
      = ( vi0->normal*b0 + vi1->normal*b1 + vi2->normal*b2 ).normalized();

   // Transform back to world space
   Matrix M = _O2ITransform_inverse;
   M.transpose();
   n = M * n;
   n.w() = 1;
   return n.normalized();
}

void Object::computeNeighbourhood(
   const Point3& centre,
   float radius,
   DistanceMetric distanceMetric,
   const Vector3& normal,
   FIndex startingFace,
   list< VIndex >& neighbouringVertices,
   list< float >& vertexDistances,
   list< FIndex >& neighbouringFaces
) {
   ASSERT( neighbouringVertices.empty() && vertexDistances.empty()
      && neighbouringFaces.empty()
   );

   Point3 c = _O2ITransform_inverse * ( centre - _I2WTranslation );
   // FIXME: the normal must be converted to object space.
   // FIXME: we don't handle scaling transformations at all;
   // there's no attempt to convert the radius or vertexDistances

   float radiusSquared = radius * radius;

   // FIXME: we don't really need vector<>s, we could use regular C arrays.
   // And ideally, we would either cache these arrays, or (even better)
   // allocate them on the stack.  What we do here probably causes
   // allocation on the heap, which is slow.
   // Allocation on the stack may not be possible.
   // On the other hand, caching the arrays would require us to
   // re-initialize them each time (e.g. with a memset());
   // but that's still probably faster than heap allocation + initialization.
   vector< bool > visitedVertices( _vertices.size(), false );
   vector< bool > visitedFaces( _faces.size(), false );

   visitNeighbours(
      _faces[startingFace].v[0], c, radiusSquared, distanceMetric, normal,
      visitedVertices, neighbouringVertices, vertexDistances,
      visitedFaces, neighbouringFaces
   );
   // The call to visitNeighbours() may not have caused all the
   // triangle's corners to be visited.
   if ( ! visitedVertices[ _faces[startingFace].v[1] ] )
      visitNeighbours(
         _faces[startingFace].v[1], c, radiusSquared, distanceMetric, normal,
         visitedVertices, neighbouringVertices, vertexDistances,
         visitedFaces, neighbouringFaces
      );
   if ( ! visitedVertices[ _faces[startingFace].v[2] ] )
      visitNeighbours(
         _faces[startingFace].v[2], c, radiusSquared, distanceMetric, normal,
         visitedVertices, neighbouringVertices, vertexDistances,
         visitedFaces, neighbouringFaces
      );
}

