
#include "Manipulator.h"
#include "WorkSpace.h"
#include "Camera.h"
#include "drawutil.h"
#include <GL/gl.h>


const int Manipulator::_thresholdDistanceSquared = 100;
const float Manipulator::_translationMultiplier = 2;
const float Manipulator::_rotationMultiplier = 0.5;
const float Manipulator::_scaleMultiplier = 0.5;


void Manipulator::computeCurrentAxis() {
   float d[3];
   int j = -1;
   for ( int i = 0; i < 3; ++i ) {
      d[i] = _axis[i].distanceToPointSquared( _mx, _my );
      if ( d[i] < _thresholdDistanceSquared && (j==-1 || d[i]<=d[j]) )
         j = i; // current best
   }
   _currentAxis = j;
}

Manipulator::Status Manipulator::press(
   int x, int y, bool LMB, bool MMB
) {
   if ( ! _visible ) return ERROR;
   ASSERT( LMB || MMB );
   _mx = x;
   _my = y;
   if ( ! _dragging ) {
      computeCurrentAxis();
      if ( _currentAxis == -1 ) return ERROR;
      _LMB = LMB;
      _MMB = MMB;
      _dragging = true;
      return REDRAW;
   }
   // one of the mouse buttons was already down
   ASSERT( _LMB || _MMB );
   _LMB = LMB;
   _MMB = MMB;
   return DONT_REDRAW;
}

Manipulator::Status Manipulator::move(
   int x, int y
) {
   // FIXME: unfortunately, this manipulator exhibits hysterisis.
   // If the user drags, and then drags back to the starting point,
   // the transformation is not always null.  Perhaps we should
   // save the location where the initial press occurs, and always
   // measure displacements with respect to that point.

   if ( ! _visible ) return ERROR;
   if ( ! _dragging ) {
      short oldAxis = _currentAxis;
      _mx = x;
      _my = y;
      computeCurrentAxis();
      return oldAxis != _currentAxis ? REDRAW : DONT_REDRAW;
   }
   ASSERT( _currentAxis != -1 );
   Vector3 v = _frame[ _currentAxis ];

   // FIXME: because of drift, the object's frame can become skewed.
   // This seems to happen especially when the user rotates the object
   // quickly.  This is actually a pretty serious bug, and probably
   // requires a redesign of how local-world transform matrices are
   // stored in each object.
   v = v.normalized();

   float magnitude
      = _size * _axis[ _currentAxis ].scaledComponent( x-_mx, y-_my );
   _mx = x;
   _my = y;
   if ( _LMB && _MMB ) _workSpace.scaleSelection(
      Vector3(1,1,1) + v*(_scaleMultiplier*magnitude),
      _origin, _onlyTransformLocalSpace
   );
   else if ( _LMB ) _workSpace.translateSelection(
      v*(_translationMultiplier*magnitude), _onlyTransformLocalSpace
   );
   else if ( _MMB ) _workSpace.rotateSelection(
      _rotationMultiplier*magnitude, v, _origin, _onlyTransformLocalSpace
   );

   return REDRAW;
}

Manipulator::Status Manipulator::release(
   int x, int y, bool LMB, bool MMB
) {
   if ( ! _visible ) return ERROR;
   _mx = x;
   _my = y;
   _LMB = LMB;
   _MMB = MMB;
   if ( ! _LMB && ! _MMB && _dragging ) {
      _dragging = false;
      computeCurrentAxis();
      return REDRAW;
   }
   return DONT_REDRAW;
}

void Manipulator::draw() {

   ASSERT( _camera != 0 );

   const list< OIndex >& selectedObjects( _workSpace.getSelectedObjects() );
   if ( selectedObjects.empty() ) {
      _visible = false;
      return;
   }
   _visible = true;
   bool usingLocalSpace = false;
   if ( selectedObjects.size() > 1 ) {
      _origin = _workSpace.getSelectionCentre();
      _frame[0] = Vector3( 1, 0, 0 );
      _frame[1] = Vector3( 0, 1, 0 );
      _frame[2] = Vector3( 0, 0, 1 );
   }
   else {
      const Object * object
         = _workSpace.getScene().getObject( selectedObjects.front() );
      _origin = object->getLocalOrigin();
      if ( _useLocalSpace ) {
         usingLocalSpace = true;
         object->getLocalFrame( _frame[0], _frame[1], _frame[2] );
      }
      else {
         _frame[0] = Vector3( 1, 0, 0 );
         _frame[1] = Vector3( 0, 1, 0 );
         _frame[2] = Vector3( 0, 0, 1 );
      }
   }

   float z = _camera->computePixel( _origin, _axis[0].A_x, _axis[0].A_y );
   _size = _camera->convertLength( z, 0.4 );
   _axis[2] = _axis[1] = _axis[0];
   _camera->computePixel(_origin+(_frame[0]*_size),_axis[0].B_x,_axis[0].B_y);
   _camera->computePixel(_origin+(_frame[1]*_size),_axis[1].B_x,_axis[1].B_y);
   _camera->computePixel(_origin+(_frame[2]*_size),_axis[2].B_x,_axis[2].B_y);

   if ( _onlyTransformLocalSpace ) {
      glLineStipple( 3, 0xAAAA );
      glEnable( GL_LINE_STIPPLE );
   }
   Point3 colour1, colour2;
   if ( usingLocalSpace ) {
      colour1 = Point3( 0, 0.5, 0 );
      colour2 = Point3( 0, 1, _dragging ? 1 : 0 );
   }
   else {
      colour1 = Point3( 0.5, 0, 0 );
      colour2 = Point3( 1, _dragging ? 1 : 0, 0 );
   }
   drawFrame(
      _origin, _frame[0], _frame[1], _frame[2], _size, true, true,
      _currentAxis == 0 ? colour2 : colour1,
      _currentAxis == 1 ? colour2 : colour1,
      _currentAxis == 2 ? colour2 : colour1
   );
   if ( _onlyTransformLocalSpace ) {
      glDisable( GL_LINE_STIPPLE );
   }
   if ( ! usingLocalSpace ) {
      glColor3f( 0.5, 0.5, 0.5 );
      list< OIndex >::const_iterator it = selectedObjects.begin();
      for ( ; it != selectedObjects.end(); ++it ) {
         _workSpace.getScene().getObject( *it )
            ->draw( false, false, false, true, 0.5 * _size );
      }
   }
}

