
#include "Camera.h"
#include <GL/gl.h>


// A good angle for the field-of-view is 30 degrees.
// Setting this very small will give the user the impression
// of looking through a telescope.
// Setting it very big will give the impression of a wide-angle
// lens, with strong foreshortening effects that are especially
// noticeable near the edges of the viewport.
const float Camera::_initialFieldOfViewSemiAngle = M_PI * 15 / 180.0;

// This should be a little greater than 1.0.
// If it's set to one, the user will initially
// just *barely* see all of the scene in the window
// (assuming the scene is a sphere of radius scene_radius).
const float Camera::_fudgeFactor = 1.1f;

// This should be a little greater than 0.0.
// If no minimum feature size is given by the user,
//    minimum_feature_size = _nearPlaneFactor * scene_radius
const float Camera::_nearPlaneFactor = 0.1f;

// far_plane = _farPlaneFactor * scene_radius
const float Camera::_farPlaneFactor = 15;

// Used for yaw, pitch, roll.
const float Camera::_rotationSpeedInDegreesPerRadius = 150;

// Used for orbit.
const float Camera::_orbitingSpeedInDegreesPerRadius = 300;

// The target (or point-of-interest) of the camera is kept
// at a minimal distance of
//   _pushThreshold * near_plane
// from the camera position.
// This number should be greater than 1.0, to ensure
// that the target is never clipped.
const float Camera::_pushThreshold = 1.3f;


void Camera::initialize(
   const Point3& scene_centre, float scene_radius,
   int window_width_in_pixels, int window_height_in_pixels,
   int viewport_x1, int viewport_y1, int viewport_x2, int viewport_y2
) {
   _initialGround = Vector3( 0, 1, 0 );
   ASSERT_IS_NORMALIZED( _initialGround );

   setSceneCentre( scene_centre );
   setSceneRadius( scene_radius );
   _minimum_feature_size = _nearPlaneFactor * _scene_radius;
   _near_plane = _minimum_feature_size;

   _window_width_in_pixels = window_width_in_pixels;
   _window_height_in_pixels = window_height_in_pixels;
   _viewport_x1 = viewport_x1;
   _viewport_y1 = viewport_y1;
   _viewport_x2 = viewport_x2;
   _viewport_y2 = viewport_y2;

   reset();
   resizeViewport(
      window_width_in_pixels, window_height_in_pixels,
      viewport_x1, viewport_y1, viewport_x2, viewport_y2
   );
}

void Camera::computeViewportWidthAndHeight() {

   // recompute _viewport_width, _viewport_height
   //
   int viewport_width_in_pixels = _viewport_x2 - _viewport_x1 + 1;
   int viewport_height_in_pixels = _viewport_y2 - _viewport_y1 + 1;
   if ( viewport_width_in_pixels < viewport_height_in_pixels ) {
      _viewport_width = 2.0 * _viewport_radius;
      _viewport_height = _viewport_width
         * (viewport_height_in_pixels-1) / (float)(viewport_width_in_pixels-1);
   }
   else {
      _viewport_height = 2.0 * _viewport_radius;
      _viewport_width = _viewport_height
         * (viewport_width_in_pixels-1) / (float)(viewport_height_in_pixels-1);
   }
}

void Camera::setSceneRadius( float r ) {
   ASSERT( r > 0 );
   if ( r <= 0 ) return;
   _scene_radius = r;
   _far_plane = _farPlaneFactor * _scene_radius;
}

void Camera::setMinimumFeatureSize( float s ) {
   ASSERT( s > 0 );
   if ( s <= 0 ) return;
   _minimum_feature_size = s;

   ASSERT( _near_plane > 0 );
   float k = _minimum_feature_size / _near_plane;
   _near_plane = _minimum_feature_size;
   _viewport_radius *= k;
   computeViewportWidthAndHeight();
}

void Camera::reset() {

   double tangent = tan( _initialFieldOfViewSemiAngle );

   _viewport_radius = _near_plane * tangent;
   float distance_from_target = _fudgeFactor * _scene_radius / tangent;

   computeViewportWidthAndHeight();

   // The ground vector can change if the user rolls the camera,
   // hence we reset it here.
   _ground = _initialGround;

   _up = _ground;

   Vector3 v;
   if ( _up.x()==0 && _up.y()==1 && _up.z()==0 )
      v = Vector3( 0, 0, 1 );
   else v = _up.choosePerpendicular().normalized();

   _target = _scene_centre;
   _position = _target + v * distance_from_target;
}

void Camera::resizeViewport(
   int window_width_in_pixels, int window_height_in_pixels,
   int viewport_x1, int viewport_y1,
   int viewport_x2, int viewport_y2
) {
   ASSERT( viewport_x1 < viewport_x2 );
   ASSERT( viewport_y1 < viewport_y2 );

   _window_width_in_pixels = window_width_in_pixels;
   _window_height_in_pixels = window_height_in_pixels;
   _viewport_x1 = viewport_x1;
   _viewport_y1 = viewport_y1;
   _viewport_x2 = viewport_x2;
   _viewport_y2 = viewport_y2;
   _viewport_cx = ( _viewport_x1 + _viewport_x2 ) * 0.5;
   _viewport_cy = ( _viewport_y1 + _viewport_y2 ) * 0.5;

   float viewport_radius_x = _viewport_cx - _viewport_x1;
   float viewport_radius_y = _viewport_cy - _viewport_y1;
   _viewport_radius_in_pixels = ( viewport_radius_x < viewport_radius_y )
      ? viewport_radius_x
      : viewport_radius_y;

   computeViewportWidthAndHeight();

   // This function expects coordinates relative to an origin
   // in the lower left corner of the window (i.e. y+ is up).
   // Also, the dimensions passed in should be between pixel centres.
   //
   // Say you have a window composed of the pixels marked with '.',
   // and want a viewport over the pixels marked with 'x':
   //
   //    ............
   //    ............
   //    .xxxxxxx....
   //    .xxxxxxx....
   //    .xxxxxxx....
   //    .xxxxxxx....
   //    .xxxxxxx....
   //    ............
   //    ............
   //    ............
   //
   // You would have to call glViewport( 1, 3, 6, 4 );
   //
   glViewport(
      _viewport_x1,
      _window_height_in_pixels - 1 - _viewport_y2,
      _viewport_x2 - _viewport_x1,
      _viewport_y2 - _viewport_y1
   );
}

void Camera::transform() {

   glMatrixMode( GL_PROJECTION );
   glLoadIdentity();  // clear matrix

   ASSERT( _near_plane < _far_plane );

   glFrustum(
      - 0.5 * _viewport_width,  0.5 * _viewport_width,    // left, right
      - 0.5 * _viewport_height, 0.5 * _viewport_height,   // bottom, top
      _near_plane, _far_plane
   );

   Matrix m;
   m.setToLookAt( _position, _target, _up );
   glMultMatrixf( m.get() );
}

void Camera::zoomIn( float delta_pixels ) {

   // this should be a little greater than 1.0
   static const float magnificationFactorPerPixel = 1.005;

   _viewport_radius *= pow( magnificationFactorPerPixel, - delta_pixels );

   computeViewportWidthAndHeight();
}

void Camera::orbit(
   float old_x_pixels, float old_y_pixels,
   float new_x_pixels, float new_y_pixels
) {
   float pixelsPerDegree = _viewport_radius_in_pixels
      / _orbitingSpeedInDegreesPerRadius;
   float radiansPerPixel = 1.0 / pixelsPerDegree * M_PI / 180;

   Vector3 t2p = _position - _target;

   Matrix m;
   m.setToRotation(
      (old_x_pixels-new_x_pixels) * radiansPerPixel,
      _ground
   );
   t2p = m*t2p;
   _up = m*_up;
   Vector3 right = (_up ^ t2p).normalized();
   m.setToRotation(
      (old_y_pixels-new_y_pixels) * radiansPerPixel,
      right
   );
   t2p = m*t2p;
   _up = m*_up;
   _position = _target + t2p;
}

void Camera::orbit(
   float old_x_pixels, float old_y_pixels,
   float new_x_pixels, float new_y_pixels,
   const Point3& centre
) {
   float pixelsPerDegree = _viewport_radius_in_pixels
      / _orbitingSpeedInDegreesPerRadius;
   float radiansPerPixel = 1.0 / pixelsPerDegree * M_PI / 180;

   Vector3 t2p = _position - _target;
   Vector3 c2p = _position - centre;

   Matrix m;
   m.setToRotation(
      (old_x_pixels-new_x_pixels) * radiansPerPixel,
      _ground
   );
   c2p = m*c2p;
   t2p = m*t2p;
   _up = m*_up;
   Vector3 right = (_up ^ t2p).normalized();
   m.setToRotation(
      (old_y_pixels-new_y_pixels) * radiansPerPixel,
      right
   );
   c2p = m*c2p;
   t2p = m*t2p;
   _up = m*_up;
   _position = centre + c2p;
   _target = _position - t2p;
}

void Camera::translateSceneRightAndUp(
   float delta_x_pixels, float delta_y_pixels
) {
   Vector3 direction = _target - _position;
   float distance_from_target = direction.length();
   direction = direction.normalized();

   float translationSpeedInUnitsPerRadius =
      distance_from_target * _viewport_radius / _near_plane;
   float pixelsPerUnit = _viewport_radius_in_pixels
      / translationSpeedInUnitsPerRadius;

   Vector3 right = direction ^ _up;

   Vector3 translation
      = right * ( - delta_x_pixels / pixelsPerUnit )
      + _up * ( - delta_y_pixels / pixelsPerUnit );

   _position += translation;
   _target += translation;
}

void Camera::dollyCameraForward( float delta_pixels, bool pushTarget ) {
   Vector3 direction = _target - _position;
   float distance_from_target = direction.length();
   direction = direction.normalized();

   float translationSpeedInUnitsPerRadius =
      distance_from_target * _viewport_radius / _near_plane;
   float pixelsPerUnit = _viewport_radius_in_pixels
      / translationSpeedInUnitsPerRadius;

   float dollyDistance = delta_pixels / pixelsPerUnit;

   if ( ! pushTarget ) {
      distance_from_target -= dollyDistance;
      if ( distance_from_target < _pushThreshold*_near_plane ) {
         distance_from_target = _pushThreshold*_near_plane;
      }
   }

   _position += direction * dollyDistance;
   _target = _position + direction * distance_from_target;
}

void Camera::pitchCameraUp( float delta_pixels ) {

   float pixelsPerDegree = _viewport_radius_in_pixels
      / _rotationSpeedInDegreesPerRadius;
   float angle = delta_pixels / pixelsPerDegree * M_PI / 180;

   Vector3 p2t = _target - _position;

   Vector3 right = (p2t.normalized()) ^ _up;

   Matrix m;
   m.setToRotation( angle, right );
   p2t = m*p2t;
   _up = m*_up;
   _target = _position + p2t;
}

void Camera::yawCameraRight( float delta_pixels ) {

   float pixelsPerDegree = _viewport_radius_in_pixels
      / _rotationSpeedInDegreesPerRadius;
   float angle = delta_pixels / pixelsPerDegree * M_PI / 180;

   Vector3 p2t = _target - _position;

   Matrix m;
   m.setToRotation( -angle, _ground );
   p2t = m*p2t;
   _up = m*_up;
   _target = _position + p2t;
}

void Camera::rollCameraRight( float delta_pixels ) {

   float pixelsPerDegree = _viewport_radius_in_pixels
      / _rotationSpeedInDegreesPerRadius;
   float angle = delta_pixels / pixelsPerDegree * M_PI / 180;

   Vector3 direction = (_target - _position).normalized();

   Matrix m;
   m.setToRotation( angle, direction );
   _ground = m*_ground;
   _up = m*_up;
}

void Camera::lookAt( const Point3& p ) {
   _target = p;
   Vector3 direction = (_target - _position).normalized();
   Vector3 right = ( direction ^ _ground ).normalized();
   _up = right ^ direction;
   ASSERT_IS_NORMALIZED( _up );
}

Ray Camera::computeRay( int pixel_x, int pixel_y ) const {
   // this is a point on the near plane, in camera space
   Point3 p(
      (pixel_x-_viewport_cx)*_viewport_radius/_viewport_radius_in_pixels,
      (_viewport_cy-pixel_y)*_viewport_radius/_viewport_radius_in_pixels,
      _near_plane
   );

   // transform p to world space
   Vector3 direction = (_target - _position).normalized();
   Vector3 right = direction ^ _up;
   Vector3 v = right*p.x() + _up*p.y() + direction*p.z();
   p = _position + v;

   return Ray( p, v.normalized() );
}

float Camera::computePixel(
   const Point3& p, int& pixel_x, int& pixel_y
) const {
   // Transform the point from world space to camera space.

   Vector3 direction = (_target - _position).normalized();
   Vector3 right = direction ^ _up;

   // Note that (right, _up, direction) form an orthonormal basis.
   // To transform a point from camera space to world space,
   // we can use the 3x3 matrix formed by concatenating the
   // 3 vectors written as column vectors.  The inverse of such
   // a matrix is simply its transpose.  So here, to convert from
   // world space to camera space, we do

   Vector3 v = p-_position;
   float x = v*right;
   float y = v*_up;
   float z = v*direction;

   // (or, more simply, the projection of a vector onto a unit vector
   // is their dot product)

   float k = _near_plane / z;

   pixel_x = (int)(
      k*_viewport_radius_in_pixels*x/_viewport_radius + _viewport_cx + 0.5f
   );
   pixel_y = (int)(
      _viewport_cy - k*_viewport_radius_in_pixels*y/_viewport_radius + 0.5f
   );
   return z;
}

float Camera::convertLength(
   float z_distance, float fractionOfViewportSize
) const {
   return z_distance * fractionOfViewportSize * 2
      * _viewport_radius / _near_plane; //tangent of field-of-view's semi-angle
}

