
#ifndef MATHUTIL_H
#define MATHUTIL_H


#include "global.h"
#include <vector>
#include <math.h>


class Point3;
class Vector3;
Vector3 operator+( const Vector3&, const Vector3& );
Vector3 operator-( const Point3&, const Point3& );
Point3 operator+( const Point3&, const Vector3& );


// Why are the below vector and point classes homogeneous
// (i.e. they have a 4th component) ?
// Admittedly, the 4th component is usually equal to 1,
// and thus wastes CPU time during computations.
// However, having all points and vectors homogeneous
// makes it simpler to work with 4x4 matrices,
// and it also guarantees well-defined behaviour when
// we call a gl*() routine and pass in vector.get()
// or point.get(), whether the routine expects an array
// of 3 components or 4.
// For example, glLightfv(GL_LIGHT0,GL_POSITION,point.get())
// expects an array of 4 elements to be passed in as the 3rd argument.
// If you pass in an array of 3 elements, you can get strange behaviour
// that can take a long time to track down.


class Vector3 {
private:
   float _v[4];
public:
   Vector3() { _v[0] = _v[1] = _v[2] = 0;  _v[3] = 1; }
   Vector3( float x, float y, float z, float w = 1 ) {
      _v[0] = x;  _v[1] = y;  _v[2] = z;  _v[3] = w;
   }

   // copy
   Vector3( const Vector3& v ) {
      _v[0] = v.x();  _v[1] = v.y();  _v[2] = v.z();  _v[3] = v.w();
   }
   Vector3& operator=( const Vector3& v ) {
      _v[0] = v.x();  _v[1] = v.y();  _v[2] = v.z();  _v[3] = v.w();
      return *this;
   }

   // type conversion
   explicit Vector3( const Point3& );

   float x() const { return _v[0]; }
   float y() const { return _v[1]; }
   float z() const { return _v[2]; }
   float w() const { return _v[3]; }
   float& x() { return _v[0]; }
   float& y() { return _v[1]; }
   float& z() { return _v[2]; }
   float& w() { return _v[3]; }

   // used to pass coordinates directly to OpenGL routines
   const float * get() const { return _v; }

   float lengthSquared() const { return x()*x() + y()*y() + z()*z(); }

   double length() const { return sqrt( lengthSquared() ); }

   Vector3 normalized() const;

   Vector3 choosePerpendicular() const;

   // Returns one of 0,1,2, for x,y,z respectively.
   // (If there is a tie, the smaller index is returned.)
   // Can be interpreted as the axis-aligned plane that
   // is most parallel to the vector
   // (e.g. if 0 is returned, then the yz plane is most parallel).
   // Note that the returned value can be used to index into
   // the array returned by get().
   short indexOfLeastComponent() const;

   // Returns one of 0,1,2, for x,y,z respectively.
   // (If there is a tie, the smaller index is returned.)
   // Can be interpreted as the axis-aligned plane that
   // is most perpendicular to the vector
   // (e.g. if 0 is returned, then the yz plane is most perpendicular).
   // Note that the returned value can be used to index into
   // the array returned by get().
   short indexOfGreatestComponent() const;

   inline Vector3 operator-() const {
      return Vector3( -x(), -y(), -z(), w() );
   }
   Vector3 operator*( float k ) const {
      return Vector3( k*x(), k*y(), k*z() );
   }
   Vector3 operator+=( const Vector3& v ) {
      _v[0] += v.x();  _v[1] += v.y();  _v[2] += v.z();
      ASSERT( v.w() == 1 );
      return *this;
   }
};


class Point3 {
private:
   float _p[4];
public:
   Point3() { _p[0] = _p[1] = _p[2] = 0;  _p[3] = 1; }
   Point3( float x, float y, float z, float w = 1 ) {
      _p[0] = x;  _p[1] = y;  _p[2] = z;  _p[3] = w;
   }

   // copy
   Point3( const Point3& p ) {
      _p[0] = p.x();  _p[1] = p.y();  _p[2] = p.z();  _p[3] = p.w();
   }
   Point3& operator=( const Point3& p ) {
      _p[0] = p.x();  _p[1] = p.y();  _p[2] = p.z();  _p[3] = p.w();
      return *this;
   }

   // type conversion
   explicit Point3( const Vector3& );

   float x() const { return _p[0]; }
   float y() const { return _p[1]; }
   float z() const { return _p[2]; }
   float w() const { return _p[3]; }
   float& x() { return _p[0]; }
   float& y() { return _p[1]; }
   float& z() { return _p[2]; }
   float& w() { return _p[3]; }

   // used to pass coordinates directly to OpenGL routines
   const float * get() const { return _p; }

   Point3 operator+=( const Vector3& v ) {
      _p[0] += v.x();  _p[1] += v.y();  _p[2] += v.z();
      ASSERT( v.w() == 1 );
      return *this;
   }
   Point3 operator-=( const Vector3& v ) {
      _p[0] -= v.x();  _p[1] -= v.y();  _p[2] -= v.z();
      ASSERT( v.w() == 1 );
      return *this;
   }
};


class Matrix {
private:
   float _m[16];
public:
   Matrix() { setToIdentity(); }
   void setToIdentity();

   // Apparently, there is no need to define a
   // copy constructor or assignment operator

   float operator()( int i ) const { return _m[i]; }
   float& operator()( int i ) { return _m[i]; }

   // used to pass coordinates directly to OpenGL routines
   const float * get() const { return _m; }

   void transpose();

   void setToTranslation( const Vector3& );

   void setToScale( const Vector3& );

   // The angle must be in [-pi,pi], and corresponds to a counterclockwise
   // rotation around v, with v pointing up at the observer.
   // Also, v must be normalized.
   void setToRotation( float angle_in_radians, const Vector3& v );

   void setToLookAt(
      const Point3& eye, const Point3& target, const Vector3& up,
      bool inverted = false
   );
};


class Ray {
public:
   Point3 origin;
   Vector3 direction;  // a unit vector
   Ray() {}
   Ray( const Point3& o, const Vector3& d ) : origin(o), direction(d) {
      ASSERT_IS_NORMALIZED( direction );
   }
   Point3 point( float t ) const { return origin + direction * t; }
};


class AlignedBox {
private:
   bool _isEmpty;
   Point3 _p0, _p1;
public:
   AlignedBox() : _isEmpty( true ) {}
   AlignedBox( const Point3& min, const Point3& max ) :
      _isEmpty( false ), _p0( min ), _p1( max )
   {
      ASSERT( _p0.x() <= _p1.x() );
      ASSERT( _p0.y() <= _p1.y() );
      ASSERT( _p0.z() <= _p1.z() );
   }
   void clear() { _isEmpty = true; }
   void bound( const Point3& );
   void bound( const vector< Point3 >&, const Matrix& );
   void bound( const AlignedBox& );

   bool isEmpty() const { return _isEmpty; }
   bool contains( const Point3& p ) {
      return
         _p0.x() <= p.x() && p.x() <= _p1.x()
         && _p0.y() <= p.y() && p.y() <= _p1.y()
         && _p0.z() <= p.z() && p.z() <= _p1.z();
   }
   const Point3& getMin() const { return _p0; }
   const Point3& getMax() const { return _p1; }
   Vector3 getDiagonal() const { return _p1 - _p0; }
   Point3 getCentre() const {
      return Point3( (Vector3(_p0)+Vector3(_p1))*0.5f );
   }
   Point3 getCorner( int i ) const {
      return Point3(
         i & 1 ? _p0.x() : _p1.x(),
         i & 2 ? _p0.y() : _p1.y(),
         i & 4 ? _p0.z() : _p1.z()
      );
   }

   // returns false if no intersection;
   // returns true if there *may* be an intersection.
   bool intersects( const Ray& ) const;
};


inline Vector3::Vector3( const Point3& p ) { 
   _v[0] = p.x();  _v[1] = p.y();  _v[2] = p.z();  _v[3] = p.w();
}
inline Point3::Point3( const Vector3& v ) {
   _p[0] = v.x();  _p[1] = v.y();  _p[2] = v.z();  _p[3] = v.w();
}
inline Vector3 operator+( const Vector3& v1, const Vector3& v2 ) {
   ASSERT( v1.w() == 1 && v2.w() == 1 );
   return Vector3( v1.x()+v2.x(), v1.y()+v2.y(), v1.z()+v2.z() );
}
inline float operator*( const Vector3& v1, const Vector3& v2 ) {
   ASSERT( v1.w() == 1 && v2.w() == 1 );
   return v1.x()*v2.x() + v1.y()*v2.y() + v1.z()*v2.z();
}
inline Point3 operator+( const Point3& p, const Vector3& v ) {
   return Point3( p.x()+v.x(), p.y()+v.y(), p.z()+v.z() );
}
inline Vector3 operator-( const Point3& p1, const Point3& p2 ) {
   return Vector3( p1.x()-p2.x(), p1.y()-p2.y(), p1.z()-p2.z() );
}
inline Point3 operator-( const Point3& p, const Vector3& v ) {
   return Point3( p.x()-v.x(), p.y()-v.y(), p.z()-v.z() );
}
inline Vector3 operator^( const Vector3& v1,  const Vector3& v2 ) {
   return Vector3(
      v1.y()*v2.z() - v1.z()*v2.y(),
      v1.z()*v2.x() - v1.x()*v2.z(),
      v1.x()*v2.y() - v1.y()*v2.x()
   );
}
Vector3 operator*( const Matrix&, const Vector3& );
inline Point3 operator*( const Matrix& m, const Point3& p ) {
   return Point3( m * Vector3( p ) );
}
Matrix operator*( const Matrix&, const Matrix& );


class LineSegment {
public:
   int A_x, A_y, B_x, B_y;
   float distanceToPointSquared( int P_x, int P_y ) const;

   // Computes the component of the given vector in the
   // direction of the line segment, and divides by the
   // length of the line segment.
   // Equivalent to the dot product divided by the length squared
   // of the line segment, i.e. (B-A)*(dx,dy) / |(B-A)|^2
   float scaledComponent( int dx, int dy ) const;
};


#endif /* MATHUTIL_H */

