
#include "Brush.h"
#include "Manipulator.h"
#include "WorkSpace.h"
#include "Camera.h"
#include "drawutil.h"
#include "SerialMouse.h"

#include <GL/glut.h>


// On an O2, try /dev/ttyd1 or /dev/ttyd2.
// On a linux box, try /dev/ttyS0 or /dev/ttyS1.
char serialPort[] = "/dev/ttyS0";
SerialMouse secondMouse;
int secondMouse_x, secondMouse_y;
int timerInterval = 1000/30;  // in milliseconds

#define FILTER_SECOND_MOUSE_DURING_TWO_HANDED_CAMERA_NAVIGATION

// These factors are for converting motion of the second mouse
// to the corresponding motion of the primary mouse.
static const float SECOND_TO_FIRST_X_FACTOR = 1.28/2.5;
static const float SECOND_TO_FIRST_Y_FACTOR = 1/2.0;

Camera* camera = 0;
static const float minSceneRadius = 8;
WorkSpace workspace;
Manipulator manipulator( workspace );
Brush brush( workspace );
Tool * currentTool = & manipulator;
bool toolActive = false;

Scene::DrawMode drawMode = Scene::FLAT_SHADED;
bool cullBackfaces = true;
bool drawTarget = false;
Ray pickingRay;
bool drawPickingRay = false;
bool twoHandedCameraNavigation = false;

bool LMB=false, MMB=false, RMB=false;
bool CTRL=false, ALT=false, SHIFT=false;
int mouse_x, mouse_y, old_mouse_x, old_mouse_y;
bool pollPrimaryMouse = false;

// MI stands for Menu Item
#define MI_CREATE_TETRAHEDRON 11
#define MI_CREATE_CUBE 12
#define MI_CREATE_OCTAHEDRON 13
#define MI_CREATE_FLAT_RECTANGULAR_MESH 14
#define MI_CREATE_BIG_FLAT_RECTANGULAR_MESH 15
#define MI_CREATE_SPHERE 16
#define MI_CREATE_SEASHELL 17
#define MI_CYCLE_DRAW_MODE 20
#define MI_TOGGLE_DISPLAY_OF_WORLD_SPACE_AXES 31
#define MI_TOGGLE_DISPLAY_OF_CAMERA_TARGET 32
#define MI_TOGGLE_DISPLAY_OF_BACKFACES 33
#define MI_TOGGLE_DISPLAY_OF_PICKING_RAY 34
#define MI_LOOKAT_SELECTION 41
#define MI_TOGGLE_TRANSFORMATION_SPACE 42
#define MI_TOGGLE_TRANSFORMED_THING 43
#define MI_TOGGLE_TOOL 44
#define MI_DELETE_SELECTION 45
#define MI_TOGGLE_2_HANDED_CAMERA_NAVIGATION 50
#define MI_RESET_CAMERA 60
#define MI_QUIT 70


void updateCamera( bool allowShrinking = false ) {
   static const float SQRT1_3 = 1.0/sqrt(3.0);
   static float currentSceneRadius = 0;

   AlignedBox box = workspace.getScene().getBoundingBox();

   // we like the region around the origin to always be visible
   float d = minSceneRadius * SQRT1_3;
   AlignedBox box2( Point3(-d,-d,-d), Point3(d,d,d) );
   box.bound( box2 );

   float newSceneRadius = box.getDiagonal().length() / 2;
   if ( newSceneRadius > currentSceneRadius || allowShrinking ) {
      currentSceneRadius = newSceneRadius;
      camera->setSceneRadius( newSceneRadius );
   }
   camera->setSceneCentre( box.getCentre() );
}

void updateBrush( float delta_x, float delta_y ) {
   brush.increaseAmplitude( 5 * delta_y );
   brush.increaseRadius( - 3 * delta_x );
   // FIXME: we should probably cause a paint() too
}

void reshapeCallback( int width, int height ) {
   if ( 0 != camera )
      camera->resizeViewport( width, height );
}

void drawCallback() {
   glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

   glDepthFunc( GL_LEQUAL );
   glEnable( GL_DEPTH_TEST );

   if ( cullBackfaces ) glEnable( GL_CULL_FACE );
   else glDisable( GL_CULL_FACE );

   if ( drawMode != Scene::WIREFRAME ) {
      Point3 lightPosition = camera->getPosition()
         + ( camera->getPosition() - camera->getTarget() );
      glLightfv( GL_LIGHT0, GL_POSITION, lightPosition.get() );
      glEnable( GL_LIGHT0 );
      // don't call glEnable( GL_LIGHTING ) -- it'll be done for us
   }

   camera->transform();

   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();

   workspace.draw( drawMode );
   currentTool->draw();

   glColor3f( 0, 1, 1 );

   if ( drawPickingRay )
      drawRay( pickingRay );

   // draw little crosshairs at the camera's target
   if ( drawTarget ) {
      const Point3& t = camera->getTarget();
      glColor3f( 0, 1, 1 );
      drawCrossHairs( t, camera->convertLength( t, 0.05 ) );
   }

   glutSwapBuffers();
}

void mouseCallback( int button, int state, int x, int y ) {
   int mod = glutGetModifiers();
   CTRL = mod & GLUT_ACTIVE_CTRL;
   ALT = mod & GLUT_ACTIVE_ALT;
   SHIFT = mod & GLUT_ACTIVE_SHIFT;

   switch ( button ) {
      case GLUT_LEFT_BUTTON : LMB = state == GLUT_DOWN; break;
      case GLUT_MIDDLE_BUTTON : MMB = state == GLUT_DOWN; break;
      case GLUT_RIGHT_BUTTON : RMB = state == GLUT_DOWN; break;
   }
   old_mouse_x = mouse_x;
   old_mouse_y = mouse_y;
   mouse_x = x;
   mouse_y = y;
   if ( pollPrimaryMouse ) return;

   Tool::Status s;
   if ( toolActive ) {
      if ( state == GLUT_DOWN ) {
         s = currentTool->press( x, y, LMB, MMB );
      }
      else {
         /* s = */ currentTool->release( x, y, LMB, MMB );
         if ( !LMB && !MMB ) toolActive = false;
         updateCamera();
         // force a redraw to make sure the camera's far plane is correct
         s = Tool::REDRAW;
      }
      if ( s == Tool::REDRAW )
         glutPostRedisplay();
   }
   else if ( state == GLUT_DOWN && !(CTRL || ALT) ) {
      s = currentTool->press( x, y, LMB, MMB );
      if ( s == Tool::ERROR ) {
         if ( button == GLUT_LEFT_BUTTON ) {
            // no part of the manip was selected, so try selecting object(s)
            pickingRay = camera->computeRay( x, y );
            list< OIndex > objectList;
            workspace.getScene().intersectAll( pickingRay, objectList, true );
            if ( SHIFT )
               workspace.xorSelection( objectList );
            else
               workspace.setSelection( objectList );
            glutPostRedisplay();
         }
      }
      else {
         toolActive = true;
         if ( s == Tool::REDRAW )
            glutPostRedisplay();
      }
   }
}

void passiveMotionCallback( int x, int y ) {
   if ( currentTool->move(x,y) == Tool::REDRAW )
      glutPostRedisplay();
   old_mouse_x = mouse_x;
   old_mouse_y = mouse_y;
   mouse_x = x;
   mouse_y = y;
}

void motionCallback( int x, int y ) {

   if ( ! pollPrimaryMouse ) {
      int delta_x = x - mouse_x;
      int delta_y = mouse_y - y;
      int delta = delta_x + delta_y;

      if ( toolActive ) {
         if ( currentTool->move(x,y) == Tool::REDRAW )
            glutPostRedisplay();
      }
      else if ( CTRL && ALT ) {
         if ( LMB )
            updateBrush( delta_x, delta_y );
      }
      else if ( CTRL ) {
         if ( LMB && MMB )
            camera->rollCameraRight( delta );
         else if ( LMB ) {
            camera->pitchCameraUp( delta_y );
            camera->yawCameraRight( delta_x );
         }
         else if ( MMB )
            camera->zoomIn( delta );
      }
      else if ( ALT ) {
         if ( LMB && MMB )
            camera->dollyCameraForward( 3*delta, false );
         else if ( LMB )
            camera->orbit( mouse_x, mouse_y, x, y );
         else if ( MMB )
            camera->translateSceneRightAndUp( delta_x, delta_y );
      }

      glutPostRedisplay();
   }
   old_mouse_x = mouse_x;
   old_mouse_y = mouse_y;
   mouse_x = x;
   mouse_y = y;
}

void menuCallback( int menuItem ) {
   switch ( menuItem ) {
      case MI_CREATE_TETRAHEDRON :
         workspace.insertTetrahedron();
         updateCamera();
         glutPostRedisplay();
         break;
      case MI_CREATE_CUBE :
         workspace.insertCube();
         updateCamera();
         glutPostRedisplay();
         break;
      case MI_CREATE_OCTAHEDRON :
         workspace.insertOctahedron();
         updateCamera();
         glutPostRedisplay();
         break;
      case MI_CREATE_FLAT_RECTANGULAR_MESH :
         workspace.insertFlatRectangularMesh( 20, 20 );
         updateCamera();
         glutPostRedisplay();
         break;
      case MI_CREATE_BIG_FLAT_RECTANGULAR_MESH :
         workspace.insertFlatRectangularMesh( 50, 50 );
         updateCamera();
         glutPostRedisplay();
         break;
      case MI_CREATE_SPHERE :
         workspace.insertSphere( 50, 50 );
         updateCamera();
         glutPostRedisplay();
         break;
      case MI_CREATE_SEASHELL :
         workspace.insertSeashell();
         updateCamera();
         glutPostRedisplay();
         break;
      case MI_CYCLE_DRAW_MODE :
         drawMode
            = (Scene::DrawMode)( ( drawMode + 1 ) % Scene::NUM_DRAW_MODES );
         glutPostRedisplay();
         break;
      case MI_TOGGLE_DISPLAY_OF_WORLD_SPACE_AXES :
         workspace.toggleDisplayOfAxes();
         glutPostRedisplay();
         break;
      case MI_TOGGLE_DISPLAY_OF_CAMERA_TARGET :
         drawTarget = !drawTarget;
         glutPostRedisplay();
         break;
      case MI_TOGGLE_DISPLAY_OF_BACKFACES :
         cullBackfaces = ! cullBackfaces;
         glutPostRedisplay();
         break;
      case MI_TOGGLE_DISPLAY_OF_PICKING_RAY :
         drawPickingRay = ! drawPickingRay;
         glutPostRedisplay();
         break;
      case MI_LOOKAT_SELECTION :
         camera->lookAt( workspace.getSelectionCentre() );
         glutPostRedisplay();
         break;
      case MI_TOGGLE_TRANSFORMATION_SPACE :
         manipulator.toggleTransformationSpace();
         glutPostRedisplay();
         break;
      case MI_TOGGLE_TRANSFORMED_THING :
         manipulator.toggleWhatsBeingTransformed();
         glutPostRedisplay();
         break;
      case MI_TOGGLE_TOOL :
         if ( ! toolActive ) {
            if ( currentTool == &manipulator ) currentTool = &brush;
            else currentTool = &manipulator;
            /* Tool::Status s = */ currentTool->move( mouse_x, mouse_y );
            glutPostRedisplay();
         }
         break;
      case MI_DELETE_SELECTION :
         workspace.deleteSelection();
         updateCamera();
         glutPostRedisplay();
         break;
      case MI_TOGGLE_2_HANDED_CAMERA_NAVIGATION :
         old_mouse_x = mouse_x;
         old_mouse_y = mouse_y;
         twoHandedCameraNavigation = ! twoHandedCameraNavigation;
         pollPrimaryMouse = twoHandedCameraNavigation;
         break;
      case MI_RESET_CAMERA :
         updateCamera( true );
         camera->reset();
         glutPostRedisplay();
         break;
      case MI_QUIT :
         exit(0);
         break;
      default:
         printf("unknown menu item %d was selected\n", menuItem );
         break;
   }
}

void keyboardCallback( unsigned char key, int x, int y ) {
   switch ( key ) {
      case ' ':
         menuCallback( MI_TOGGLE_TOOL );
         break;
      case 'a':
         menuCallback( MI_TOGGLE_DISPLAY_OF_WORLD_SPACE_AXES );
         break;
      case 'b':
         menuCallback( MI_TOGGLE_DISPLAY_OF_BACKFACES );
         break;
      case 'c':
         menuCallback( MI_TOGGLE_2_HANDED_CAMERA_NAVIGATION );
         break;
      case 'd':
         menuCallback( MI_CYCLE_DRAW_MODE );
         break;
      case 'l':
         menuCallback( MI_LOOKAT_SELECTION );
         break;
      case 'p':
         menuCallback( MI_TOGGLE_TRANSFORMED_THING );
         break;
      case 'r':
         menuCallback( MI_RESET_CAMERA );
         break;
      case 's':
         menuCallback( MI_TOGGLE_TRANSFORMATION_SPACE );
         break;
      case 't':
         menuCallback( MI_TOGGLE_DISPLAY_OF_CAMERA_TARGET );
         break;
      case 27: // Escape
      //case 'q':
         menuCallback( MI_QUIT );
         break;
      case 127: // delete
         menuCallback( MI_DELETE_SELECTION );
         break;
      default:
         printf("untrapped key %d\n", (int)key );
         break;
   }
}

void specialCallback( int key, int x, int y ) {
   switch ( key ) {
      case GLUT_KEY_F1 :
         menuCallback( MI_CREATE_TETRAHEDRON );
         break;
      case GLUT_KEY_F2 :
         menuCallback( MI_CREATE_CUBE );
         break;
      case GLUT_KEY_F3 :
         menuCallback( MI_CREATE_OCTAHEDRON );
         break;
      case GLUT_KEY_F4 :
         menuCallback( MI_CREATE_FLAT_RECTANGULAR_MESH );
         break;
      case GLUT_KEY_F5 :
         menuCallback( MI_CREATE_BIG_FLAT_RECTANGULAR_MESH );
         break;
      case GLUT_KEY_F6 :
         menuCallback( MI_CREATE_SPHERE );
         break;
      default:
         printf("untrapped key %d\n", (int)key );
         break;
   }
}

void timerCallback( int /* id */ ) {

#ifdef FILTER_SECOND_MOUSE_DURING_TWO_HANDED_CAMERA_NAVIGATION
   // FIXME: the code for filtering input events should be
   // moved to the SerialMouse class.

   static const int FILTER_SIZE = 5; // set to 1 for no filtering
   static float filterSamples[ FILTER_SIZE ][2] = { 0,0 };
   int j;
#endif

   float delta_x = 0, delta_y = 0;
   if ( twoHandedCameraNavigation ) {
      static const float k = 3;
      bool redisplay = false;

      // check if the primary mouse moved
      delta_x = mouse_x - old_mouse_x;
      delta_y = old_mouse_y - mouse_y;
      if ( delta_x != 0 || delta_y != 0 ) {
         if ( LMB ) {
            redisplay = true;
            camera->pitchCameraUp( k*delta_y );
            camera->yawCameraRight( k*delta_x );
         }
         old_mouse_x = mouse_x;
         old_mouse_y = mouse_y;
      }

      // check if the secondary mouse moved
#ifdef FILTER_SECOND_MOUSE_DURING_TWO_HANDED_CAMERA_NAVIGATION
      secondMouse.poll();
      {
#else
      if ( secondMouse.poll() ) {
#endif
         delta_x = ( secondMouse.x - secondMouse_x )*SECOND_TO_FIRST_X_FACTOR;
         delta_y = ( secondMouse_y - secondMouse.y )*SECOND_TO_FIRST_Y_FACTOR;

#ifdef FILTER_SECOND_MOUSE_DURING_TWO_HANDED_CAMERA_NAVIGATION
         // shift the filter samples down one
         // FIXME: we should use a circular buffer so we could avoid shifting.
         for ( j = 0; j < FILTER_SIZE - 1; ++j ) {
            filterSamples[ j ][0] = filterSamples[ j+1 ][0];
            filterSamples[ j ][1] = filterSamples[ j+1 ][1];
         }

         // add the new sample to the end of the filter
         filterSamples[ FILTER_SIZE-1 ][0] = delta_x;
         filterSamples[ FILTER_SIZE-1 ][1] = delta_y;

         // compute the filtered (i.e. averaged) value
         delta_x = delta_y = 0;
         for ( j = 0; j < FILTER_SIZE; ++j ) {
            delta_x += filterSamples[ j ][0];
            delta_y += filterSamples[ j ][1];
         }
         delta_x /= FILTER_SIZE;
         delta_y /= FILTER_SIZE;
#endif

         if ( delta_x != 0 || delta_y != 0 ) {
            redisplay = true;
            camera->translateSceneRightAndUp( - k * delta_x, - k * delta_y );
            secondMouse_x = secondMouse.x;
            secondMouse_y = secondMouse.y;
         }
      }

      if ( redisplay ) glutPostRedisplay();
   }
   else {
      if ( secondMouse.poll() ) {
         delta_x = ( secondMouse.x - secondMouse_x )*SECOND_TO_FIRST_X_FACTOR;
         delta_y = ( secondMouse_y - secondMouse.y )*SECOND_TO_FIRST_Y_FACTOR;
         if ( secondMouse.leftBtn && secondMouse.middleBtn ) {
            camera->dollyCameraForward( 3*(delta_x+delta_y), false );
         }
         else if ( secondMouse.leftBtn ) {
            camera->orbit(
               delta_x, - delta_y,
               currentTool == &brush
                  ? brush.getCentreForOrbiting() : camera->getTarget()
            );
         }
         else if ( secondMouse.middleBtn ) {
            camera->translateSceneRightAndUp( delta_x, delta_y );
         }
         else if ( secondMouse.rightBtn ) {
            updateBrush( delta_x, delta_y );
         }
         glutPostRedisplay();
         secondMouse_x = secondMouse.x;
         secondMouse_y = secondMouse.y;
      }
   }
   glutTimerFunc( timerInterval, timerCallback, 0 );
}

int main( int argc, char *argv[] ) {

#ifdef DEBUG
#if 0
   printf( "Each vertex requires %d bytes of storage,\n"
           "and each face requires %d bytes.\n",
           sizeof( Point3 ) + sizeof( VertexInfo ),
           sizeof( Triangle )
   );
#endif
#endif

   if ( ! secondMouse.init( serialPort ) ) {
      puts("Couldn't initialize serial mouse.");
   }
   else {
      secondMouse_x = secondMouse.x;
      secondMouse_y = secondMouse.y;
      glutTimerFunc( timerInterval, timerCallback, 0 );
   }

   glutInit( &argc, argv );
   glutInitDisplayMode( GLUT_RGB | GLUT_DEPTH | GLUT_DOUBLE );
   glutInitWindowSize( 512, 512 );
   glutCreateWindow("SceneEditor");

   glutDisplayFunc( drawCallback );
   glutReshapeFunc( reshapeCallback );
   glutKeyboardFunc( keyboardCallback );
   glutSpecialFunc( specialCallback );
   glutMouseFunc( mouseCallback );
   glutPassiveMotionFunc( passiveMotionCallback );
   glutMotionFunc( motionCallback );

   // create a menu
   int mainMenu = glutCreateMenu( menuCallback );
      int creationMenu = glutCreateMenu( menuCallback );
      glutSetMenu( creationMenu ); // make it the current menu
      glutAddMenuEntry( "Tetrahedron (F1)", MI_CREATE_TETRAHEDRON );
      glutAddMenuEntry( "Cube (F2)", MI_CREATE_CUBE );
      glutAddMenuEntry( "Octahedron (F3)", MI_CREATE_OCTAHEDRON );
      glutAddMenuEntry( "Plane (F4)", MI_CREATE_FLAT_RECTANGULAR_MESH );
      glutAddMenuEntry( "Big Plane (F5)", MI_CREATE_BIG_FLAT_RECTANGULAR_MESH );
      glutAddMenuEntry( "Sphere (F6)", MI_CREATE_SPHERE );
      // FIXME glutAddMenuEntry( "Seashell", MI_CREATE_SEASHELL );
   glutSetMenu( mainMenu ); // make it the current menu
   glutAddSubMenu( "Create...", creationMenu );
   glutAddMenuEntry( "Cycle Display Mode (d)", MI_CYCLE_DRAW_MODE );
      int displayToggleMenu = glutCreateMenu( menuCallback );
      glutSetMenu( displayToggleMenu );
      glutAddMenuEntry( "World Space Axes (a)",
         MI_TOGGLE_DISPLAY_OF_WORLD_SPACE_AXES );
      glutAddMenuEntry( "Camera Target (t)",
         MI_TOGGLE_DISPLAY_OF_CAMERA_TARGET );
      glutAddMenuEntry( "Backfaces (b)", MI_TOGGLE_DISPLAY_OF_BACKFACES );
      glutAddMenuEntry( "Picking Ray", MI_TOGGLE_DISPLAY_OF_PICKING_RAY );
   glutSetMenu( mainMenu );
   glutAddSubMenu( "Toggle Display of...", displayToggleMenu );
      int selectionMenu = glutCreateMenu( menuCallback );
      glutSetMenu( selectionMenu );
      glutAddMenuEntry( "Look At (l)", MI_LOOKAT_SELECTION );
      glutAddMenuEntry( "Local/World Space (s)",
         MI_TOGGLE_TRANSFORMATION_SPACE );
      glutAddMenuEntry( "Geometry/Pivot (p)", MI_TOGGLE_TRANSFORMED_THING );
      glutAddMenuEntry( "Transform/Brush (space)", MI_TOGGLE_TOOL );
      glutAddMenuEntry( "Delete (del)", MI_DELETE_SELECTION );
   glutSetMenu( mainMenu ); 
   glutAddSubMenu( "Selection", selectionMenu );
   glutAddMenuEntry( "Toggle 2-Handed Cam Nav (c)",
      MI_TOGGLE_2_HANDED_CAMERA_NAVIGATION );
   glutAddMenuEntry( "Reset Camera (r)", MI_RESET_CAMERA );
   glutAddMenuEntry( "Quit (Esc)", MI_QUIT );
   glutAttachMenu( GLUT_RIGHT_BUTTON );//attach the menu to the current window

   int width = glutGet( GLUT_WINDOW_WIDTH );
   int height = glutGet( GLUT_WINDOW_HEIGHT );
   camera = new Camera(
      width, height,
      minSceneRadius,  // the scene is initially empty
      Point3( 0, 0, 0) // the scene is initially empty
   );
   manipulator.setCamera( camera );
   brush.setCamera( camera );

   glutMainLoop();
}

