
#include "Brush.h"
#include "WorkSpace.h"
#include "drawutil.h"
#include <GL/gl.h>


// This should be in (0,1), and close to zero.
const float Brush::_cutoff = 0.05;

const float Brush::_ln_cutoff = log( Brush::_cutoff );


float Brush::decayFunction( float r ) const {
   float x = r / _radius;
   return _amplitude * exp( _ln_cutoff * x*x );
}

float Brush::inverseDecayFunction( float y ) const {
   return _radius * sqrt( log(y/_amplitude) / _ln_cutoff );
}

void Brush::computeIntersection() {
   _ray = _camera->computeRay( _mx, _my );
   _isThereAnIntersection = false;
   if ( _dragging ) {
      // try intersecting with the saved object
      ASSERT( _savedObject != 0 );
      if ( _savedObject->getBoundingBox().intersects( _ray ) ) {
         if ( _savedObject->intersectNearest( _ray, _currentFace, _t, true ) ) {
            _intersection = _ray.point( _t );
            _normal = _savedObject->getNormal( _currentFace, _intersection );
            _isThereAnIntersection = true;
            _coveredVertices.clear();
            _distances.clear();
            _coveredFaces.clear();
            _savedObject->computeNeighbourhood(
               _intersection, _radius, Object::CYLINDRICAL_DISTANCE, _normal,
               _currentFace, _coveredVertices, _distances, _coveredFaces
            );
         }
      }
   }
   else {
      OIndex objectIndex;
      if (
         _workSpace.getScene().intersectNearest(
            _ray, objectIndex, _currentFace, _t, true
         )
         && _workSpace.isSelected( objectIndex )
      ) {
         _currentObject = _workSpace.getScene().getObject( objectIndex );
         _intersection = _ray.point( _t );
         _normal = _currentObject->getNormal( _currentFace, _intersection );
         _isThereAnIntersection = true;
      }
      else _currentObject = 0;
   }
}

void Brush::paint( bool noRecursion ) {
   ASSERT( _dragging );
   if ( ! _isThereAnIntersection ) return;
   ASSERT( _currentObject != 0 );
   float sign = 1;
   // FIXME: what should LMB+MMB do ?  flatten the mesh ?
   if ( _MMB ) sign = -1;  //FIXME: should we also draw the brush upside down ?
   list< VIndex >::const_iterator it = _coveredVertices.begin();
   list< float >::const_iterator it2 = _distances.begin();
   for ( ; it != _coveredVertices.end(); ++it, ++it2 ) {
      float newDisplacement = sign * decayFunction( *it2 );
      if ( sign * newDisplacement > sign * _displacement[*it] )
         _displacement[*it] = newDisplacement;
//Vector3 vvv = _savedObject->getVertexNormal(*it);
//printf("vvv");PRINT( vvv );
      _currentObject->setVertex(
         *it,
         _savedObject->getVertex(*it)
         + _savedObject->getVertexNormal(*it) * _displacement[*it]
      );
   }
   if ( noRecursion ) return;
   list< FIndex > facesToSplit;
   _currentObject->testFaces( _coveredFaces, _thresholdArea, facesToSplit );
   if ( ! facesToSplit.empty() ) {
      // make the displacement array grow
      _displacement.insert( _displacement.end(), facesToSplit.size(), 0 );

      _savedObject->subdivideFaces( facesToSplit );
      _currentObject->subdivideFaces( facesToSplit );

      computeIntersection();   // FIXME: wasteful (?)
      paint( true );
   }
}

Brush::Status Brush::press(
   int x, int y, bool LMB, bool MMB
) {
   ASSERT( LMB || MMB );
   _mx = x;
   _my = y;
   if ( ! _dragging ) {
      computeIntersection();
      if ( ! _isThereAnIntersection ) return ERROR;
      ASSERT( _currentObject != 0 && _savedObject == 0 );
      _LMB = LMB;
      _MMB = MMB;
      _dragging = true;
      _savedObject = new Object();
      *_savedObject = *_currentObject;
      _displacement.clear();  // FIXME: is this necessary ?
      _displacement.resize( _savedObject->getNbVertices(), 0 );
      computeIntersection();  // FIXME: wasteful hack to get coveredVertices
      _thresholdArea = _savedObject->getMaxFaceArea();
printf("t == %f\n", _thresholdArea );
   }
   else {
      // one of the mouse buttons was already down
      ASSERT( _LMB || _MMB );
      _LMB = LMB;
      _MMB = MMB;
   }
   paint();
   return REDRAW;
}

Brush::Status Brush::move(
   int x, int y
) {
   if ( ! _dragging ) {
      bool wasThereAnyIntersection = _isThereAnIntersection;
      _mx = x;
      _my = y;
      computeIntersection();
      return ( wasThereAnyIntersection || _isThereAnIntersection )
         ? REDRAW : DONT_REDRAW;
   }
// FIXME: we should loop over intermediate values of _mx,_my and paint for each
   _mx = x;
   _my = y;
   computeIntersection();
   paint();
   return REDRAW;
}

Brush::Status Brush::release(
   int x, int y, bool LMB, bool MMB
) {
   _mx = x;
   _my = y;
   _LMB = LMB;
   _MMB = MMB;
   if ( ! _LMB && ! _MMB && _dragging ) {
      delete _savedObject;  _savedObject = 0;
      _dragging = false;
      computeIntersection();
      return REDRAW;
   }
   return DONT_REDRAW;
}

void Brush::draw() {

   ASSERT( _camera != 0 );
   if ( _dragging ) {
      ASSERT( _savedObject != 0 );
      // FIXME: instead of drawing the whole object, it would be
      // preferable to draw only triangles that have moved.
      // This would require us to maintain an array of boolean
      // flags over the triangles, and update it in paint().
      // It would also require a new version of Object::draw()
      // that accepts an array of booleans.
      glColor3f( 0.5, 0.5, 0.5 );
//      _savedObject->draw();

      if ( _isThereAnIntersection ) {
         //glColor3f( 1, 1, 0 );
         // FIXME: do we really want to draw this ?
         list< VIndex >::const_iterator it = _coveredVertices.begin();
         for ( ; it != _coveredVertices.end(); ++it ) {
            Point3 p = _savedObject->getVertex( *it );
            drawCrossHairs( p, 0.3 );
         }
      }
   }
   if ( _isThereAnIntersection ) {
#if 1
      glColor3f( 1, 0, 0 );
      static const int N = 5;
      static const float alpha = 0.01;  // ensures top circle is not point
      float y_0 = _amplitude * _cutoff;
      float delta_y = ( _amplitude - y_0 - alpha*_amplitude ) / N;
      for ( int i = 0; i <= N; ++i ) {
         float y = i * delta_y + y_0;
         float r = inverseDecayFunction( y );
         drawCircle(
            _intersection + _normal*y, _normal, r,
            // to save a bit on rendering time, we multiply by 2
            // yielding approximately one polygon edge per 2 pixels
            2 * _camera->convertPixelLength( _intersection, 1 )
         );
      }
#else
      glColor3f( 1, 1, 0 );
      drawCircle(
         _intersection, _normal,
         _radius,
         // to save a bit on rendering time, we multiply by 2
         // yielding approximately one polygon edge per 2 pixels
         2 * _camera->convertPixelLength( _intersection, 1 )
      );
#endif
   }
}

