
#include "global.h"
#include <GL/gl.h>


// ================================================== MATRIX OPERATIONS

void assignToUnity( float matrix[ 16 ] ) {

   matrix[ 0] = 1.0; matrix[ 1] = 0.0; matrix[ 2] = 0.0; matrix[ 3] = 0.0;
   matrix[ 4] = 0.0; matrix[ 5] = 1.0; matrix[ 6] = 0.0; matrix[ 7] = 0.0;
   matrix[ 8] = 0.0; matrix[ 9] = 0.0; matrix[10] = 1.0; matrix[11] = 0.0;
   matrix[12] = 0.0; matrix[13] = 0.0; matrix[14] = 0.0; matrix[15] = 1.0;
}

void copyMatrix( float destination[ 16 ], float source[ 16 ] ) {

   int j;
   for (j = 0; j < 16; ++j) {
      destination[ j ] = source[ j ];
   }
}

void transposeMatrix( float matrix[ 16 ] ) {

   float tmp;
   #define SWAP(a,b) (tmp)=(a); (a)=(b); (b)=(tmp);
   SWAP(matrix[ 1],matrix[ 4]);
   SWAP(matrix[ 2],matrix[ 8]);
   SWAP(matrix[ 3],matrix[12]);
   SWAP(matrix[ 7],matrix[13]);
   SWAP(matrix[11],matrix[14]);
   SWAP(matrix[ 6],matrix[ 9]);
   #undef SWAP
}

void assignToTranslation(
   float matrix[16],
   float delta_x, float delta_y, float delta_z
) {
   glMatrixMode( GL_MODELVIEW );
   glLoadIdentity();  // clear matrix
   glTranslatef( delta_x, delta_y, delta_z );
   glGetFloatv( GL_MODELVIEW_MATRIX, matrix ); // save into matrix
}

void assignToRotation(
   float matrix[16],
   float angle_in_degrees,
   float axis_x, float axis_y, float axis_z
) {
   glMatrixMode( GL_MODELVIEW );
   glLoadIdentity();  // clear matrix
   glRotatef( angle_in_degrees, axis_x, axis_y, axis_z );
   glGetFloatv( GL_MODELVIEW_MATRIX, matrix ); // save into matrix
}

void premultiplyWithTranslation(
   float matrix[16],
   float delta_x, float delta_y, float delta_z
) {
   glMatrixMode( GL_MODELVIEW );
   glLoadIdentity();  // clear matrix
   glTranslatef( delta_x, delta_y, delta_z );
   glMultMatrixf( matrix );
   glGetFloatv( GL_MODELVIEW_MATRIX, matrix ); // save into matrix
}

void premultiplyWithRotation(
   float matrix[16], 
   float angle_in_degrees,
   float axis_x, float axis_y, float axis_z
) {
   glMatrixMode( GL_MODELVIEW );
   glLoadIdentity();  // clear matrix
   glRotatef( angle_in_degrees, axis_x, axis_y, axis_z );
   glMultMatrixf( matrix );
   glGetFloatv( GL_MODELVIEW_MATRIX, matrix ); // save into matrix
}

// ================================================== HIERARCHICAL MODELING

// Assumptions: the caller wants to end up in a local space
// where z+ is up, y+ is forward, and x+ is right.
//
void pushLevel(
   float position_x, float position_y, float position_z,
   float direction_x, float direction_y, float direction_z,
   bool isReverse
) {
   glPushMatrix();

   // Rotate the direction vector onto the y+ axis
   //
   float hypotenuse = sqrt(
      direction_x * direction_x + direction_y * direction_y
   );
   float azimuth = asin( direction_x / hypotenuse );
   if ( direction_y < 0.0 ) {
      azimuth = M_PI - azimuth;
   }
   float elevation = atan( direction_z / hypotenuse );

   if ( isReverse ) {
      glRotatef( - 180.0 * elevation / M_PI, 1.0, 0.0, 0.0 );
      glRotatef( 180.0 * azimuth / M_PI, 0.0, 0.0, 1.0 );
      glTranslatef( - position_x, - position_y, - position_z );
   }
   else {
      glTranslatef( position_x, position_y, position_z );
      glRotatef( - 180.0 * azimuth / M_PI, 0.0, 0.0, 1.0 );
      glRotatef( 180.0 * elevation / M_PI, 1.0, 0.0, 0.0 );
   }
}

void popLevel() {

   glPopMatrix();
}

// ================================================== DRAWING ROUTINES

void drawSphere( float x, float y, float z, float radius, LevelOfDetail LOD ) {

   pushLevel( x, y, z, 1.0, 0.0, 0.0, false );
   if ( HIGH_LOD != LOD ) {
      float R = radius;
      glBegin(GL_TRIANGLE_STRIP);
         glNormal3f( -R, 0.0, 0.0 );
         glVertex3f( -R, 0.0, 0.0 );
         glNormal3f( 0.0, -R, 0.0 );
         glVertex3f( 0.0, -R, 0.0 );
         glNormal3f( 0.0, 0.0, R );
         glVertex3f( 0.0, 0.0, R );
         glNormal3f( R, 0.0, 0.0 );
         glVertex3f( R, 0.0, 0.0 );
         glNormal3f( 0.0, R, 0.0 );
         glVertex3f( 0.0, R, 0.0 );
         glNormal3f( 0.0, 0.0, -R );
         glVertex3f( 0.0, 0.0, -R );
         glNormal3f( -R, 0.0, 0.0 );
         glVertex3f( -R, 0.0, 0.0 );
         glNormal3f( 0.0, -R, 0.0 );
         glVertex3f( 0.0, -R, 0.0 );
      glEnd();
      glBegin(GL_TRIANGLES);
         glNormal3f( 0.0, 0.0, R );
         glVertex3f( 0.0, 0.0, R );
         glNormal3f( 0.0, R, 0.0 );
         glVertex3f( 0.0, R, 0.0 );
         glNormal3f( -R, 0.0, 0.0 );
         glVertex3f( -R, 0.0, 0.0 );

         glNormal3f( 0.0, 0.0, -R );
         glVertex3f( 0.0, 0.0, -R );
         glNormal3f( R, 0.0, 0.0 );
         glVertex3f( R, 0.0, 0.0 );
         glNormal3f( 0.0, -R, 0.0 );
         glVertex3f( 0.0, -R, 0.0 );
      glEnd();
   }
   else {

      // this should never change
      //
      static const int numHemispheres = 2;

      // any of these can be increased to make the sphere smoother
      //
      static const int numStrips = 4;
      static const int numberOfLatitudinalPatchesPerHemisphere = 5;
      static const int numberOfLongitudinalPatchesPerStrip = 5;

      // This structure contains the mesh geometry of the northern
      // (or upper) half of one "strip" of a unit sphere.
      // (A "strip" here means a lune with (1/numStrips)th
      // the area of the sphere's surface; where a lune is any
      // region on the sphere's surface delimited by two great circles.
      // Note that with strips, the great circles intersect at the
      // north and south pole of the sphere, and are thus cut in half
      // by the equator.)
      //
      static struct {
         GLfloat vertex[ 3 ];
      } vertices[ 1 + numberOfLatitudinalPatchesPerHemisphere ]
                [ 1 + numberOfLongitudinalPatchesPerStrip ];
      static bool haveVerticesBeenInitialized = false;

      int j, k;

      // This block generates the mesh the first time it's needed.
      //
      if ( ! haveVerticesBeenInitialized ) {
         for ( j = 0; j <= numberOfLatitudinalPatchesPerHemisphere; ++j ) {
            float latitude
               = j * 0.5 * M_PI / numberOfLatitudinalPatchesPerHemisphere;
            float z = sin( latitude );
            float horizontal = cos( latitude );
            for ( k = 0; k <= numberOfLongitudinalPatchesPerStrip; ++k ) {
               float longitude = k *
                  2.0 * M_PI / numberOfLongitudinalPatchesPerStrip / numStrips;
               float x = horizontal * cos( longitude );
               float y = horizontal * sin( longitude );
               vertices[ j ][ k ].vertex[ 0 ] = x;
               vertices[ j ][ k ].vertex[ 1 ] = y;
               vertices[ j ][ k ].vertex[ 2 ] = z;
            }
         }
         haveVerticesBeenInitialized = true;
      }

      // In a GL_TRIANGLE_STRIP block, if the vertices sent in are
      // numbered 0,1,2,3,... the resulting strip will look like
      //
      //     0---2---4---6---8--- ...
      //      \ / \ / \ / \ / \
      //       1---3---5---7---9 ...
      //
      // and will be cowposed of triangles (0,1,2), (2,1,3), (2,3,4), ...
      // The ordering is important.  By convention, polygons whose vertices
      // appear in counterclockwise order on the screen are front-facing.
      // If you create polygons with vertices in the wrong order, and
      // back-face culling is turned on, you'll never see the polygons
      // appear on the screen.


      // glFrontFace(GL_CW);  // we're going to use the opposite convention

      int hemisphere, strip;
      float x, y, z;

      for (hemisphere = 0; hemisphere < numHemispheres; ++hemisphere) {
         glPushMatrix();
         glRotatef(hemisphere*180.0,0.0,1.0,0.0);
         for (strip = 0; strip < numStrips; ++strip) {
            glPushMatrix();
            glRotatef(
               (hemisphere == 0 ? -strip : strip+1)*360.0/numStrips,
               0.0, 0.0, 1.0
            );

            for (j = 0; j < numberOfLatitudinalPatchesPerHemisphere; ++j) {
               glBegin(GL_TRIANGLE_STRIP);
                  for (k = 0; k <= numberOfLongitudinalPatchesPerStrip; ++k) {
                     x = radius * vertices[ j+1 ][ k ].vertex[0];
                     y = radius * vertices[ j+1 ][ k ].vertex[1];
                     z = radius * vertices[ j+1 ][ k ].vertex[2];

                     glNormal3fv( vertices[ j+1 ][ k ].vertex );
                     glVertex3f( x, y, z);

                     x = radius * vertices[ j ][ k ].vertex[ 0 ];
                     y = radius * vertices[ j ][ k ].vertex[ 1 ];
                     z = radius * vertices[ j ][ k ].vertex[ 2 ];

                     glNormal3fv( vertices[ j ][ k ].vertex );
                     glVertex3f( x, y, z );
                  }
               glEnd();
            }
            glPopMatrix();
         }
         glPopMatrix();
      }
   }
   popLevel();
}

