
#include "TrainTrackSwitch.h"
#include <GL/gl.h>


static const GLfloat trainTrackSwitchColour[] = { 1.0, 1.0, 0.0, 1.0 };


void TrainTrackSwitch::computeLength() {

   float direction_x, direction_y, direction_z;

   if ( _isInStateA ) {
      _trackA1->computePositionAndDirection(
         _endA1 == FINAL_END_POINT ? _trackA1->getLength() : 0.0,
         _start_x, _start_y, _start_z,
         direction_x, direction_y, direction_z
      );
      _trackA2->computePositionAndDirection(
         _endA2 == FINAL_END_POINT ? _trackA2->getLength() : 0.0,
         _end_x, _end_y, _end_z,
         direction_x, direction_y, direction_z
      );
   }
   else {
      _trackB1->computePositionAndDirection(
         _endB1 == FINAL_END_POINT ? _trackB1->getLength() : 0.0,
         _start_x, _start_y, _start_z,
         direction_x, direction_y, direction_z
      );
      _trackB2->computePositionAndDirection(
         _endB2 == FINAL_END_POINT ? _trackB2->getLength() : 0.0,
         _end_x, _end_y, _end_z,
         direction_x, direction_y, direction_z
      );
   }

   float delta_x, delta_y, delta_z;

   delta_x = _end_x - _start_x;
   delta_y = _end_y - _start_y;
   delta_z = _end_z - _start_z;

   _length = sqrt( delta_x * delta_x + delta_y * delta_y + delta_z * delta_z );
   _start_direction_x = delta_x / _length;
   _start_direction_y = delta_y / _length;
   _start_direction_z = delta_z / _length;

   ASSERT_IS_NORMALIZED(
      _start_direction_x, _start_direction_y, _start_direction_z
   );
}

void TrainTrackSwitch::toggle() {

   if ( _isInStateA ) {
      _trackA1->disconnect( _endA1 );
      _trackA2->disconnect( _endA2 );
      _trackB1->connect( _endB1, this, INITIAL_END_POINT );
      _trackB2->connect( _endB2, this, FINAL_END_POINT );
      _isInStateA = false;
   }
   else {
      _trackB1->disconnect( _endB1 );
      _trackB2->disconnect( _endB2 );
      _trackA1->connect( _endA1, this, INITIAL_END_POINT );
      _trackA2->connect( _endA2, this, FINAL_END_POINT );
      _isInStateA = true;
   }

   computeLength();
}

void TrainTrackSwitch::draw( LevelOfDetail LOD, bool isWireFrame ) {

   glMaterialfv( GL_FRONT, GL_AMBIENT_AND_DIFFUSE, trainTrackSwitchColour );

   float hypotenuse = sqrt(
      _start_direction_x * _start_direction_x
      + _start_direction_y * _start_direction_y
   );
   // These equations can be derived using similar triangles.
   float delta_x = _start_direction_y * TRAIN_TRACK_WIDTH / hypotenuse;
   float delta_y = _start_direction_x * TRAIN_TRACK_WIDTH / hypotenuse;

   glBegin( GL_LINE_LOOP );
      glVertex3f( _start_x + 0.5*delta_x, _start_y - 0.5*delta_y, _start_z );
      glVertex3f( _end_x + 0.5*delta_x, _end_y - 0.5*delta_y, _end_z );
      glVertex3f( _start_x - 0.5*delta_x, _start_y + 0.5*delta_y, _start_z );
      glVertex3f( _end_x - 0.5*delta_x, _end_y + 0.5*delta_y, _end_z );
   glEnd();
}

void TrainTrackSwitch::computePositionAndDirection(
   float distanceAlongSwitch,
   float & x,
   float & y,
   float & z,
   float & direction_x,
   float & direction_y,
   float & direction_z
) {
   x = _start_x + _start_direction_x * distanceAlongSwitch;
   y = _start_y + _start_direction_y * distanceAlongSwitch;
   z = _start_z + _start_direction_z * distanceAlongSwitch;
   direction_x = _start_direction_x;
   direction_y = _start_direction_y;
   direction_z = _start_direction_z;

   // Ensure a unit vector is returned.
   //
   ASSERT_IS_NORMALIZED( direction_x, direction_y, direction_z );
}

void TrainTrackSwitch::getConnection(
   TrackEndPoint connectedEnd,
   AbstractTrainTrack * & trackConnectedTo,
   TrackEndPoint & endConnectedTo
) {
   if ( INITIAL_END_POINT == connectedEnd ) {
      trackConnectedTo = _isInStateA ? _trackA1 : _trackB1;
      endConnectedTo = _isInStateA ? _endA1 : _endB1;
   }
   else {
      trackConnectedTo = _isInStateA ? _trackA2 : _trackB2;
      endConnectedTo = _isInStateA ? _endA2 : _endB2;
   }
}

