
#include "TrainTrack.h"
#include "SectionOfTrainTrack.h"
#include <GL/gl.h>


static const GLfloat trainTrackColour[] = { 0.0, 1.0, 1.0, 1.0 };


TrainTrack::TrainTrack(
   float start_x,
   float start_y,
   float start_z,
   float start_direction_x,
   float start_direction_y,
   float start_direction_z
) :
   _numberOfSections( 0 ),
   _arrayOfSections( NULL ),
   _arrayOfCumulativeLengths( NULL ),

   _total_length( 0.0 ),

   _start_x( start_x ),
   _start_y( start_y ),
   _start_z( start_z ),
   _start_direction_x( start_direction_x ),
   _start_direction_y( start_direction_y ),
   _start_direction_z( start_direction_z ),

   _track1( NULL ), _track2( NULL ),
   _end1( INITIAL_END_POINT ), _end2( INITIAL_END_POINT )
{
   NORMALIZE( _start_direction_x, _start_direction_y, _start_direction_z );
}

TrainTrack::~TrainTrack() {

   for (int j = 0; j < _numberOfSections; ++j) {
      delete _arrayOfSections[ j ];
      _arrayOfSections[ j ] = NULL;
   }
   delete [] _arrayOfSections;
   _arrayOfSections = NULL;

   delete [] _arrayOfCumulativeLengths;
   _arrayOfCumulativeLengths = NULL;

   _numberOfSections = 0;
}

void TrainTrack::appendStraightSection( float length, float delta_z ) {

   ASSERT( length > delta_z );

   float start_x;
   float start_y;
   float start_z;
   float start_direction_x;
   float start_direction_y;
   float start_direction_z;

   computeStartingPositionAndDirectionForNewSection(
      start_x, start_y, start_z,
      start_direction_x, start_direction_y, start_direction_z
   );

   // Although a value for the direction's z-component has already
   // been computed, we won't use this value.  Rather, we'll
   // specially recompute the z-component as a function
   // of delta_z.
   //
   // Remember: a negative delta_z means a downward slope.
   //
   {
      // This equation can be derived using similar triangles.
      //
      start_direction_z =
         sqrt(
            start_direction_x * start_direction_x
            + start_direction_y * start_direction_y
         )
         * delta_z / sqrt( length * length - delta_z * delta_z );
   }

   // Create the new section.
   //
   SectionOfTrainTrack * section;
   section = new StraightSection(
      length,
      start_x, start_y, start_z,
      start_direction_x, start_direction_y, start_direction_z
   );

   // Append it.
   //
   appendSection( section );
}

void TrainTrack::appendCurvedSection(
   float arc_angle_in_degrees,
   float radius,
   float delta_z
) {
   float lengthAlongCurve = fabs(
      M_PI*arc_angle_in_degrees/(double)180.0 * radius
   );

   ASSERT( lengthAlongCurve > delta_z );

   float start_x;
   float start_y;
   float start_z;
   float start_direction_x;
   float start_direction_y;
   float start_direction_z;

   computeStartingPositionAndDirectionForNewSection(
      start_x, start_y, start_z,
      start_direction_x, start_direction_y, start_direction_z
   );

   // Although a value for the direction's z-component has already
   // been computed, we won't use this value.  Rather, we'll
   // specially recompute the z-component as a function
   // of delta_z.
   //
   // Remember: a negative delta_z means a downward slope.
   //
   {
      // This equation can be derived using similar triangles.
      //
      start_direction_z =
         sqrt(
            start_direction_x * start_direction_x
            + start_direction_y * start_direction_y
         )
         * delta_z
         / sqrt( lengthAlongCurve * lengthAlongCurve - delta_z * delta_z );
   }

   // Create a new section.
   //
   SectionOfTrainTrack * section;
   section = new CurvedSection(
      M_PI * arc_angle_in_degrees / (double)180.0,
      radius,
      start_x, start_y, start_z,
      start_direction_x, start_direction_y, start_direction_z
   );

   // Append it.
   //
   appendSection( section );
}

void TrainTrack::draw( LevelOfDetail LOD, bool /*isWireFrame*/ ) {

   float oldx1, oldy1, oldx2, oldy2, oldz;
   float newx1, newy1, newx2, newy2;

   glMaterialfv( GL_FRONT, GL_AMBIENT_AND_DIFFUSE, trainTrackColour );

   // Draw track lines at intervals of one unit.
   //
   float distance = 0.0;
   int j = 0;
   do {
      ASSERT( j < _numberOfSections );

      // Convert the distance along the track
      // into a distance along the section.
      //
      float distanceAlongSection = ( j > 0 )
         ? distance - _arrayOfCumulativeLengths[ j - 1 ]
         : distance;

      // Find the corresponding position and direction in 3-space.
      //
      float x, y, z;
      float direction_x, direction_y, direction_z;
      _arrayOfSections[ j ]->computePositionAndDirection(
         distanceAlongSection,
         x, y, z,
         direction_x, direction_y, direction_z
      );

      #ifdef DEBUG
      // Test the compute*() routine to see if it yields the same result.
      // We only do this in debug mode, for verification purposes.
      // Calling compute*() to get the result is actually slower than
      // using the above method.
      //
      float test_x, test_y, test_z;
      float test_direction_x, test_direction_y, test_direction_z;
      computePositionAndDirection(
         distance,
         test_x, test_y, test_z,
         test_direction_x, test_direction_y, test_direction_z
      );
      ASSERT_IS_EQUAL( test_x, x );
      ASSERT_IS_EQUAL( test_y, y );
      ASSERT_IS_EQUAL( test_z, z );
      ASSERT_IS_EQUAL( test_direction_x, direction_x );
      ASSERT_IS_EQUAL( test_direction_y, direction_y );
      ASSERT_IS_EQUAL( test_direction_z, direction_z );
      #endif

      // Draw it.
      //
      switch ( LOD ) {
      case LOW_LOD:
         glBegin(GL_POINTS);
            glVertex3f( x, y, z );
         glEnd();
         break;
      case MEDIUM_LOD:
      case HIGH_LOD:
         float hypotenuse = sqrt(
            direction_x * direction_x + direction_y * direction_y
         );
         // These equations can be derived using similar triangles.
         float delta_x = direction_y * TRAIN_TRACK_WIDTH / hypotenuse;
         float delta_y = direction_x * TRAIN_TRACK_WIDTH / hypotenuse;

         glBegin(GL_LINES);
            glVertex3f( x + 0.5 * delta_x, y - 0.5 * delta_y, z );
            glVertex3f( x - 0.5 * delta_x, y + 0.5 * delta_y, z );

            newx1 = x + 0.3 * delta_x;
            newy1 = y - 0.3 * delta_y;
            newx2 = x - 0.3 * delta_x;
            newy2 = y + 0.3 * delta_y;

            if ( 0.0 < distance ) {
               // connect this rung with the previous one
               glVertex3f( newx1, newy1, z );
               glVertex3f( oldx1, oldy1, oldz );
               glVertex3f( newx2, newy2, z );
               glVertex3f( oldx2, oldy2, oldz );
            }
         glEnd();

         oldx1 = newx1;
         oldy1 = newy1;
         oldx2 = newx2;
         oldy2 = newy2;
         oldz = z;

         break;
      }

      // Increment.
      //
      distance += (LOD == HIGH_LOD ? 1.0 : 2.0);
      if ( distance > _arrayOfCumulativeLengths[ j ] ) {
         ++ j;
      }
   } while ( distance <= _total_length );
}

void TrainTrack::computePositionAndDirection(
   float distanceAlongTrack,
   float & x,
   float & y,
   float & z,
   float & direction_x,
   float & direction_y,
   float & direction_z
) {
   ASSERT( 0.0 <= distanceAlongTrack && distanceAlongTrack <= _total_length );
   ASSERT( _numberOfSections > 0 );

   // Find the relevant section.
   // Use binary searching to find it quickly.
   //
   int j = _numberOfSections / 2;
   {
      int step = ( _numberOfSections > 7 ) ? _numberOfSections / 4 : 1;

      while ( 1 ) {
         if ( _arrayOfCumulativeLengths[ j ] < distanceAlongTrack ) {
            j += step;
         }
         else if (
            j > 0   &&
            _arrayOfCumulativeLengths[ j - 1 ] > distanceAlongTrack
         ) {
            j -= step;
         }
         else break;

         if (step > 1) {
            step /= 2;
         }
      }
   }
   ASSERT( 0 <= j && j < _numberOfSections );

   // Convert the distance along the track
   // into a distance along the section.
   //
   float distanceAlongSection = ( j > 0 )
      ? distanceAlongTrack - _arrayOfCumulativeLengths[ j - 1 ]
      : distanceAlongTrack;
   ASSERT( 0.0 <= distanceAlongSection );
   ASSERT( distanceAlongSection <= _arrayOfSections[ j ]->getLength() );

   // Ask the section to do the computation.
   //
   _arrayOfSections[ j ]->computePositionAndDirection(
      distanceAlongSection,
      x, y, z,
      direction_x, direction_y, direction_z
   );

   // Ensure a unit vector is returned.
   //
   ASSERT_IS_NORMALIZED( direction_x, direction_y, direction_z );
}

void TrainTrack::connect(
   TrackEndPoint endToConnect,
   AbstractTrainTrack * trackToConnectTo,
   TrackEndPoint endToConnectTo
) {
   if ( INITIAL_END_POINT == endToConnect ) {
      _track1 = trackToConnectTo;
      _end1 = endToConnectTo;
   }
   else {
      _track2 = trackToConnectTo;
      _end2 = endToConnectTo;
   }
}

void TrainTrack::disconnect( TrackEndPoint endToDisconnect ) {
   if ( INITIAL_END_POINT == endToDisconnect ) {
      _track1 = NULL;
      _end1 = INITIAL_END_POINT;
   }
   else {
      _track2 = NULL;
      _end2 = INITIAL_END_POINT;
   }
}

void TrainTrack::getConnection(
   TrackEndPoint connectedEnd,
   AbstractTrainTrack * & trackConnectedTo,
   TrackEndPoint & endConnectedTo
) {
   if ( INITIAL_END_POINT == connectedEnd ) {
      trackConnectedTo = _track1;
      endConnectedTo = _end1;
   }
   else {
      trackConnectedTo = _track2;
      endConnectedTo = _end2;
   }
}

void TrainTrack::computeStartingPositionAndDirectionForNewSection(
   float & start_x,
   float & start_y,
   float & start_z,
   float & start_direction_x,
   float & start_direction_y,
   float & start_direction_z
) {
   // Compute the starting position and direction of a new section.
   //
   if ( _numberOfSections > 0 ) {

      // The starting position and direction of a new section
      // are equal to the ending position and direction
      // of the current last section.
      //
      _arrayOfSections[ _numberOfSections - 1 ]->computePositionAndDirection(
         _arrayOfSections[ _numberOfSections - 1 ]->getLength(),
         start_x, start_y, start_z,
         start_direction_x, start_direction_y, start_direction_z
      );
   }
   else {

      // The track currently has no sections. Hence,
      // the starting position and direction of a new section
      // are equal to the starting position and direction
      // of the train track.
      //
      start_x = _start_x;
      start_y = _start_y;
      start_z = _start_z;
      start_direction_x = _start_direction_x;
      start_direction_y = _start_direction_y;
      start_direction_z = _start_direction_z;
   }
}

void TrainTrack::appendSection( SectionOfTrainTrack * section ) {

   // Make room for the new section: enlarge the arrays by one.
   //
   {
      SectionOfTrainTrack ** newArrayOfSections;
      float * newArrayOfCumulativeLengths;
      newArrayOfSections = new SectionOfTrainTrackPtr[ _numberOfSections + 1 ];
      newArrayOfCumulativeLengths = new float [ _numberOfSections + 1 ];
      for (int j = 0; j < _numberOfSections; ++j ) {
         newArrayOfSections[ j ] = _arrayOfSections[ j ];
         newArrayOfCumulativeLengths[ j ] = _arrayOfCumulativeLengths[ j ];
      }
      delete [] _arrayOfSections;
      delete [] _arrayOfCumulativeLengths;
      _arrayOfSections = newArrayOfSections;
      _arrayOfCumulativeLengths = newArrayOfCumulativeLengths;
      ++ _numberOfSections;
   }

   // Copy the new section into the last array entry.
   //
   _arrayOfSections[ _numberOfSections - 1 ] = section;
   _total_length += section->getLength();
   _arrayOfCumulativeLengths[ _numberOfSections - 1 ] = _total_length;
}

