
#include "Train.h"


const float Train::_cruisingSpeed = 20.0;
const float Train::_startingAcceleration = 0.5;
const float Train::_accelerationUnderNoPower = -1.0; // must be negative
const float Train::_breakingAcceleration = -1.5; // must be negative


Train::~Train() {

   for ( int j = 0; j < _numberOfCars; ++j ) {
      delete _arrayOfCars[ j ]->car;
      _arrayOfCars[ j ]->car = NULL;
      delete _arrayOfCars[ j ];
      _arrayOfCars[ j ] = NULL;
   }
   delete [] _arrayOfCars;
   _arrayOfCars = NULL;
   _numberOfCars = 0;
}

void Train::reset() {

   _isEngineOn = false;
   _speed = 0.0;
   _acceleration = 0.0;

   // save the current train
   //
   int tmp_numberOfCars = _numberOfCars;
   TrainCarStruct ** tmp_arrayOfCars = _arrayOfCars;

   _numberOfCars = 0;
   _arrayOfCars = NULL;
   int j;

   // create a new train
   //
   for ( j = 0; j < tmp_numberOfCars; ++j ) {
      appendCar( tmp_arrayOfCars[ j ]->car->getCarType() );
   }

   // delete the old train
   //
   for ( j = 0; j < tmp_numberOfCars; ++j ) {
      delete tmp_arrayOfCars[ j ]->car;
      tmp_arrayOfCars[ j ]->car = NULL;
      delete tmp_arrayOfCars[ j ];
      tmp_arrayOfCars[ j ] = NULL;
   }
   delete [] tmp_arrayOfCars;
}

bool Train::advance( float delta_time, bool & startingPointPassed ) {

   ASSERT( 0.0 <= delta_time );

   bool isTheTrainStillMoving = false;
   startingPointPassed = false;
   float distanceAlongTrackBefore = 0.0, distanceAlongTrackAfter = 0.0;

   if ( 0 < _numberOfCars && _arrayOfCars[ 0 ]->isAlive ) {
      if ( _arrayOfCars[ 0 ]->car->getAbstractTrainTrack() == _initial_track )
         distanceAlongTrackBefore
            = _arrayOfCars[ 0 ]->car->getDistanceAlongTrack();
   }

   // Adjust the speed of the train.
   //
   if ( 0.0 < _acceleration ) {
      // the train is speeding up
      _speed += _acceleration * delta_time;
      if ( _speed > _cruisingSpeed ) {
         // we've reached cruising speed, so stop accelerating
         _speed = _cruisingSpeed;
         _acceleration = 0.0;
      }
   }
   else if ( 0.0 > _acceleration ) {
      // the train is slowing down
      _speed += _acceleration * delta_time;
      if ( _speed < 0.0 ) {
         // we've come to a halt
         _speed = 0.0;
         _acceleration = 0.0;
      }
   }

   // Advance each of the individual cars by an appropriate distance.
   //
   float _speedOfCurrentCar = _speed;
   for ( int j = 0; j < _numberOfCars; ++j ) {
      ASSERT( NULL != _arrayOfCars[ j ] );
      if ( _arrayOfCars[ j ]->isAlive ) {
         ASSERT( NULL != _arrayOfCars[ j ]->car );
         _arrayOfCars[ j ]->car->advance( _speedOfCurrentCar * delta_time );
         if ( 0.0 != _speedOfCurrentCar )
            isTheTrainStillMoving = true;
      }
      else {
         _arrayOfCars[ j ]->speedOfNextCar
            += _accelerationUnderNoPower * delta_time;
         if ( _arrayOfCars[ j ]->speedOfNextCar < 0.0 ) {
            // the next car has come to a halt
            _arrayOfCars[ j ]->speedOfNextCar = 0.0;
         }
         _speedOfCurrentCar = _arrayOfCars[ j ]->speedOfNextCar;
      }
   }

   // Check if the train passed its starting point.
   //
   if ( 0 < _numberOfCars && _arrayOfCars[ 0 ]->isAlive ) {
      if (
         _arrayOfCars[ 0 ]->car->getAbstractTrainTrack() == _initial_track
      ) {
         distanceAlongTrackAfter
            = _arrayOfCars[ 0 ]->car->getDistanceAlongTrack();
         startingPointPassed
            = ( _initial_directionAlongTrack == TrainCar::FORWARD_DIRECTION )
            ? (
               distanceAlongTrackBefore < _initial_distanceAlongTrack &&
               distanceAlongTrackAfter >= _initial_distanceAlongTrack
            )
            : (
               distanceAlongTrackBefore > _initial_distanceAlongTrack &&
               distanceAlongTrackAfter <= _initial_distanceAlongTrack
            );
      }
   }

   return isTheTrainStillMoving || ( 0.0 != _acceleration );
}

void Train::draw( LevelOfDetail LOD, bool isWireFrame ) {

   for ( int j = 0; j < _numberOfCars; ++j ) {
      ASSERT( NULL != _arrayOfCars[ j ] );
      if ( _arrayOfCars[ j ]->isAlive ) {
         ASSERT( NULL != _arrayOfCars[ j ]->car );
         _arrayOfCars[ j ]->car->draw( LOD, isWireFrame );
      }
   }
}

void Train::appendCar( TrainCar::TrainCarType carType ) {

   switch ( carType ) {
   case TrainCar::LOCOMOTIVE :
      appendCar( carType, 0.64, 0.16, 0.16 );
      break;
   case TrainCar::FREIGHT_CAR :
      appendCar( carType, 0.0, 0.5, 0.8 );
      break;
   case TrainCar::CABOOSE :
      appendCar( carType, 1.0, 0.0, 0.0 );
      break;
   }
}

void Train::appendCar(
   TrainCar::TrainCarType carType,
   float colourRed, float colourGreen, float colourBlue
) {

   AbstractTrainTrack * track;
   float distanceAlongTrack;
   TrainCar::TrainCarDirection directionAlongTrack;

   // Figure out where on the track the new car should go.
   //
   if ( 0 < _numberOfCars ) {
      track = _arrayOfCars[ _numberOfCars - 1 ]->car->getAbstractTrainTrack();
      distanceAlongTrack
         = _arrayOfCars[ _numberOfCars - 1 ]->car->getDistanceAlongTrack();
      directionAlongTrack
         = _arrayOfCars[ _numberOfCars - 1 ]->car->getDirectionAlongTrack();
   }
   else {
      track = _initial_track;
      distanceAlongTrack = _initial_distanceAlongTrack;
      directionAlongTrack = _initial_directionAlongTrack;
   }

   // Create the new car.
   //
   TrainCarStruct * newTrainCarStruct = new TrainCarStruct;
   newTrainCarStruct->car = new TrainCar(
      carType, track, distanceAlongTrack, directionAlongTrack,
      colourRed, colourGreen, colourBlue
   );
   newTrainCarStruct->isAlive = true;
   newTrainCarStruct->speedOfNextCar = 0.0;

   // If this new car is not the first, it will have
   // to be moved back behind the previous car.
   //
   if ( 0 < _numberOfCars ) {
      float distance
         = 0.5 * _arrayOfCars[ _numberOfCars - 1 ]->car->getLength()
         + 0.5 * newTrainCarStruct->car->getLength();
      newTrainCarStruct->car->advance( - distance * 1.1 );
   }

   // Make room for the new car: enlarge the array by one.
   //
   {
      TrainCarStruct ** newArrayOfCars
         = new TrainCarStructPtr [ _numberOfCars + 1 ];
      for ( int j = 0; j < _numberOfCars; ++j ) {
         newArrayOfCars[ j ] = _arrayOfCars[ j ];
      }
      newArrayOfCars[ _numberOfCars ] = NULL;
      delete [] _arrayOfCars;
      _arrayOfCars = newArrayOfCars;
      ++ _numberOfCars;
   }

   // Add the new car to the last array entry.
   //
   _arrayOfCars[ _numberOfCars - 1 ] = newTrainCarStruct;
}

void Train::removeCar( int index ) {

   ASSERT( 0 <= index && index < _numberOfCars );
   ASSERT( NULL != _arrayOfCars[ index ] );

   if ( ! _arrayOfCars[ index ]->isAlive ) {
      // the car has already been removed
      return;
   }

   _arrayOfCars[ index ]->isAlive = false;

   // If all the cars between the 1st car and this one are
   // alive (i.e. connected), then this car was being pulled
   // along at the same speed just prior to removal.
   // Assume this is the case.
   //
   _arrayOfCars[ index ]->speedOfNextCar = _speed;

   // Now, just in case our assumption was wrong, we search
   // backwards for a dead car, and adjust the speed accordingly.
   //
   for ( int j = index - 1; j >= 0; --j ) {
      if ( ! _arrayOfCars[ j ]->isAlive ) {
         _arrayOfCars[ index ]->speedOfNextCar
            = _arrayOfCars[ j ]->speedOfNextCar;
         break;
      }
   }
}

void Train::getBoundingGeometry( Sphere * arrayOfSpheres ) {

   ASSERT( NULL != arrayOfSpheres );
   for ( int j = 0; j < _numberOfCars; ++j ) {
      if ( ! _arrayOfCars[ j ]->isAlive ) {
         // A zero-radius will the flag the collision detection
         // algorithm to ignore this sphere.
         arrayOfSpheres[ j ].radius = 0.0;
         continue;
      }

      _arrayOfCars[ j ]->car->getPosition(
         arrayOfSpheres[ j ].x, arrayOfSpheres[ j ].y, arrayOfSpheres[ j ].z
      );

      // The "bounding sphere" doesn't have to be strictly bounding,
      // cuz that would make it rather big.  It just has to bound
      // most of the train car.  So, to compute the bounding sphere's
      // radius, we multiply the length of the train car by something
      // a bit less than 0.5.
      //
      arrayOfSpheres[ j ].radius = 0.4 * _arrayOfCars[ j ]->car->getLength();
   }
}

void Train::performCollisionTest(
   int numberOfCarsForTrain1, Sphere * geometryOfTrain1,
   int numberOfCarsForTrain2, Sphere * geometryOfTrain2,
   bool * carCollisionsForTrain1,
   bool * carCollisionsForTrain2
) {
   ASSERT( NULL != geometryOfTrain1 );
   ASSERT( NULL != geometryOfTrain2 );
   ASSERT( NULL != carCollisionsForTrain1 );
   ASSERT( NULL != carCollisionsForTrain2 );

   int j, k;

   for ( j = 0; j < numberOfCarsForTrain1; ++j )
      carCollisionsForTrain1[ j ] = false;
   for ( k = 0; k < numberOfCarsForTrain2; ++k )
      carCollisionsForTrain2[ k ] = false;

   float delta_x, delta_y, delta_z, min_distance;
   for ( j = 0; j < numberOfCarsForTrain1; ++j ) {
      if ( 0.0 < geometryOfTrain1[ j ].radius ) {
         for ( k = 0; k < numberOfCarsForTrain2; ++k ) {
            if ( 0.0 < geometryOfTrain2[ k ].radius ) {
               delta_x = geometryOfTrain1[ j ].x - geometryOfTrain2[ k ].x;
               delta_y = geometryOfTrain1[ j ].y - geometryOfTrain2[ k ].y;
               delta_z = geometryOfTrain1[ j ].z - geometryOfTrain2[ k ].z;
               min_distance = geometryOfTrain1[ j ].radius
                  + geometryOfTrain2[ k ].radius;
               if (
                  delta_x*delta_x + delta_y*delta_y + delta_z*delta_z
                     < min_distance*min_distance
               ) {
                  carCollisionsForTrain1[ j ] = true;
                  carCollisionsForTrain2[ k ] = true;
               }
            }
         }
      }
   }
}

