
#include "SectionOfTrainTrack.h"


SectionOfTrainTrack::SectionOfTrainTrack(
   float length,
   float start_x,
   float start_y,
   float start_z,
   float start_direction_x,
   float start_direction_y,
   float start_direction_z
) :
   _length( length ),
   _start_x( start_x ),
   _start_y( start_y ),
   _start_z( start_z ),
   _start_direction_x( start_direction_x ),
   _start_direction_y( start_direction_y ),
   _start_direction_z( start_direction_z )
{
   NORMALIZE( _start_direction_x, _start_direction_y, _start_direction_z );
}

// ==================================================

void StraightSection::computePositionAndDirection(
   float distanceAlongTrack,
   float & x,
   float & y,
   float & z,
   float & direction_x,
   float & direction_y,
   float & direction_z
) {
   x = _start_x + _start_direction_x * distanceAlongTrack;
   y = _start_y + _start_direction_y * distanceAlongTrack;
   z = _start_z + _start_direction_z * distanceAlongTrack;
   direction_x = _start_direction_x;
   direction_y = _start_direction_y;
   direction_z = _start_direction_z;

   // Ensure a unit vector is returned.
   //
   ASSERT_IS_NORMALIZED( direction_x, direction_y, direction_z );
}

// ==================================================

CurvedSection::CurvedSection(
   float arc_angle_in_radians,
   float radius,
   float start_x,
   float start_y,
   float start_z,
   float start_direction_x,
   float start_direction_y,
   float start_direction_z
) :
   _arc_angle_in_radians( arc_angle_in_radians ),
   _radius( radius ),
   SectionOfTrainTrack(
      fabs( radius * arc_angle_in_radians ),
      start_x, start_y, start_z,
      start_direction_x, start_direction_y, start_direction_z
   )
{
   _hypotenuse = sqrt(
      _start_direction_x * _start_direction_x
      + _start_direction_y * _start_direction_y
   );

   _start_theta = - asin( _start_direction_x / _hypotenuse );
   if ( _start_direction_y < 0.0 ) {
      _start_theta = M_PI - _start_theta;
   }
   if ( _arc_angle_in_radians < 0.0 ) {
      // This curved section curves to the right.
      _start_theta += M_PI;
   }

   _arc_center_x = _start_x - _radius * cos( _start_theta );
   _arc_center_y = _start_y - _radius * sin( _start_theta );
}

void CurvedSection::computePositionAndDirection(
   float distanceAlongTrack,
   float & x,
   float & y,
   float & z,
   float & direction_x,
   float & direction_y,
   float & direction_z
) {
   float delta_theta = _arc_angle_in_radians * distanceAlongTrack / _length;
   float   sine = sin( _start_theta + delta_theta );
   float cosine = cos( _start_theta + delta_theta );

   // Remember: a negative angle means a clockwise turn.

   x = _arc_center_x + _radius * cosine;
   y = _arc_center_y + _radius * sine;
   z = _start_z + _start_direction_z * distanceAlongTrack;
   if ( _arc_angle_in_radians < 0.0 ) {
      direction_x = + _hypotenuse * sine;
      direction_y = - _hypotenuse * cosine;
   } else {
      direction_x = - _hypotenuse * sine;
      direction_y = + _hypotenuse * cosine;
   }
   direction_z = _start_direction_z;

   // Ensure a unit vector is returned.
   //
   ASSERT_IS_NORMALIZED( direction_x, direction_y, direction_z );  
}

