
#include "Rotator.h"
#include <GL/gl.h>
#include <string.h>   /* memcpy() */


void Rotator::setRotationMode( RotationMode newMode ) {

   // Save rotations of current mode so they won't be lost,
   // allowing us to transfer smoothly into the new mode.
   //
   memcpy(
      _oldRotationMatrix,   // destination
      _rotationMatrix,   // source
      16*sizeof(float)
   );

   // reset mode-specific parameters
   //
   _elevation = 0.0;
   _azimuth = 0.0;

   // set to new mode
   //
   _rotationMode = newMode;
}

void Rotator::deviceMotion( float x1, float y1, float x2, float y2 ) {

   float delta_x = x2 - x1;
   float delta_y = y2 - y1;
   float pixelsPerDegree = 0.01 * _radius;

   glMatrixMode( GL_MODELVIEW );
   glLoadIdentity();  // clear matrix

   switch ( _rotationMode ) {
      case elevationAzimuth:
         _elevation += delta_y / pixelsPerDegree;
         _azimuth += delta_x / pixelsPerDegree;
         glRotatef(_elevation,1.0,0.0,0.0);  // x+ axis points right
         glRotatef(_azimuth,0.0,1.0,0.0);    // y+ axis points up
         glMultMatrixf(_oldRotationMatrix);   // preserve previous rotations
         // do not save result in old matrix
         break;
      case aroundXYAxes:
         glRotatef(delta_y / pixelsPerDegree,1.0,0.0,0.0);
         glRotatef(delta_x / pixelsPerDegree,0.0,1.0,0.0);
         glMultMatrixf(_oldRotationMatrix);   // preserve previous rotations
         glGetFloatv( GL_MODELVIEW_MATRIX, _oldRotationMatrix );  // save result
      case virtualSphere:
      case arcball:
         float ux, uy, uz;   // previous mouse position
         float uzSquared;
         float vx, vy, vz;   // new mouse position
         float vzSquared;
         float wx, wy, wz;   // cross product (u x v)
         float dotProduct;
         float shrinkFactor;

         // compute u
         //
         ux = ( x1 - _center_x ) / _radius;
         uy = ( _center_y - y1 ) / _radius;  // uy+ axis points up, not down
         uzSquared = 1.0 - ux*ux - uy*uy;
         if ( uzSquared < 0.0 ) {
            // Shorten vector to force a magnitude of one
            shrinkFactor = sqrt( ux*ux + uy*uy );
            ux /= shrinkFactor;
            uy /= shrinkFactor;
            uz = 0.0;
         }
         else uz = sqrt( uzSquared );

         // compute v
         //
         vx = ( x2 - _center_x ) / _radius;
         vy = ( _center_y - y2 ) / _radius;
         vzSquared = 1.0 - vx*vx - vy*vy;
         if ( vzSquared < 0.0 ) {
            // Shorten vector to force a magnitude of one
            shrinkFactor = sqrt( vx*vx + vy*vy );
            vx /= shrinkFactor;
            vy /= shrinkFactor;
            vz = 0.0;
         }
         else vz = sqrt( vzSquared );

         // compute cross product w = (u x v)
         //
         wx =   uy*vz - vy*uz;
         wy = -(ux*vz - vx*uz);
         wz =   ux*vy - vx*uy;

         // Since
         //    |w| = |u||v||sin theta|
         // and
         //    |u| = |v| = 1
         // we have
         //    |w| = |sin theta|
         // But theta = asin(|w|) only if 0 <= theta <= pi/2 radians.
         // If theta > pi/2, theta != asin(|w|).
         //
         // So we calculate theta using the dot product (u.v)
         //    u.v = |u||v|(cos theta) = cos theta
         //    theta = acos(u.v)
         // This works for 0 <= theta <= pi radians.
         dotProduct = ux*vx + uy*vy + uz*vz;
         if (dotProduct < -1.0)
             dotProduct = -1.0;
         else if (dotProduct > 1.0)
             dotProduct = 1.0;
         float theta = acos(dotProduct);
         if (_rotationMode == arcball) {
            // This is the only difference between virtualSphere and arcball.
            // This magic number causes arcball to have no hysterisis.
            theta *= 2.0;
         }
         glRotatef(180 * theta / M_PI, wx, wy, wz);
         glMultMatrixf(_oldRotationMatrix);   // preserve previous rotations
         glGetFloatv(GL_MODELVIEW_MATRIX, _oldRotationMatrix);   // save result
         break;
   };

   glGetFloatv(GL_MODELVIEW_MATRIX, _rotationMatrix);   // save result
}

