
#include "Menu.h"
#include "Train.h"
#include "TrainTrackSwitch.h"
#include "TrainTrack.h"
#include "Explosion.h"
#include "Camera.h"
#include "global.h"

#include <string.h>   /* memcpy(), strcat() */
#include <stdlib.h>   /* system() */
#include <libgen.h>   /* basename() */

/* #include <X11/X.h> */
#include <X11/Xlib.h>
/* #include <X11/Xutil.h> */
#include <X11/keysym.h>

#include <GL/gl.h>
#include <GL/glx.h>   /* glX*() routines */
/* #include <GL/glu.h> */

/* #define TOGGLEABLE_WINDOW_DECORATIONS */  /* requires Motif libraries */

#ifdef TOGGLEABLE_WINDOW_DECORATIONS
#include <Xm/Xm.h>
#include <Xm/MwmUtil.h>
#endif

/* typedef enum { False = 0, True = 1 } Boolean; */
#ifndef TOGGLEABLE_WINDOW_DECORATIONS
typedef unsigned char Boolean;
#endif
typedef unsigned long Pixel;

// ==========================================================================

// global variables

struct {
   char * commandLineName;

   // Recall that in X terminology,
   // one *display* can consist of
   // many *screens*.
   //
   Display *display;
   int screen_num;
   Screen *screen_ptr;
   int screenWidth, screenHeight;      // screen dimensions in pixels
   int screenWidthMM, screenHeightMM;  // screen dimensions in millimeters
   unsigned int screenDepth;
   Window rootWindow;
   Window window;
   unsigned int windowWidth, windowHeight;   /* dimensions in pixels */
   unsigned int minOfWidthAndHeight() {
      return windowWidth < windowHeight ? windowWidth : windowHeight;
   }
   Cursor cursor;
   Pixel blackPixel, whitePixel;
   GC gc;   /* Graphics context. */
   XFontStruct *bigFont, *smallFont;
   Colormap colourMap;
} XStuff;

// display parameters
Boolean bBackFaceCulling = True;
Boolean bFrontFacing = True;
Boolean bShadingIsSmooth = True;
Boolean bDisplayText = True;
LevelOfDetail LOD = HIGH_LOD;
bool isWireframe = false;
bool isTheSceneAnimating = false;
bool simulateSoftwareFailure = false;
bool isSimulationPaused = false;
float delta_t = 0.05;  // simulation delta

static const int MENU_HEIGHT = 15;
Menu * theMenu = NULL;

static const char orbitingCameraHelpString[]
   = "LMB:orbit MMB:pan RMB:zoom Ctrl-RMB:dolly";
static const char freeFloatingCameraHelpString[]
   = "LMB:yaw/pitch MMB:pan RMB:dolly Ctrl-LMB:roll Ctrl-RMB:zoom";
const char * currentHelpString = NULL;

#ifdef TOGGLEABLE_WINDOW_DECORATIONS
Boolean bNoWindowDecorations = False;
#endif
Pixel currentTextColour;

// ==========================================================================

Pixel AllocPixelColour(double r, double g, double b) {

   XColor xcolor;

   xcolor.pixel = 0;
   xcolor.red   = (int)(r * 65535 + 0.5);  // add .5 to round-off to nearest integer
   xcolor.green = (int)(g * 65535 + 0.5);
   xcolor.blue  = (int)(b * 65535 + 0.5);
   xcolor.flags = DoRed | DoGreen | DoBlue;
   XAllocColor(XStuff.display, XStuff.colourMap, &xcolor);
   return xcolor.pixel;
}

void FreePixelColour(Pixel p) {

   XFreeColors(XStuff.display, XStuff.colourMap, &p, 1, 0);
}

// ==========================================================================

void MakeWindow(
   char * windowTitle,
   char * iconTitle,
   Pixmap iconPixmap,                      /* Pass (None) for none. */
   int minWidth, int minHeight,
   Boolean isFullScreenAndBorderless,      /* There are problems with setting this to True ... see comments below. */
   Boolean isOpenGLWindow,
   Pixel backgroundPixel,
   int argc, char *argv[]                  /* Command line arguments. */
) {
   XWMHints *wm_hints;                     /* Window manager hints. */
   XClassHint *class_hints;                /* Application class hints. */
   XSizeHints *size_hints;                 /* Preferred window geometry. */
   XTextProperty windowName,iconName;      /* Special X "strings" */
   int windowX, windowY;                   /* Upper left corner of window. */
   unsigned int windowBorderWidth;

   /* This stuff is for creating an OpenGL window. */
   GLXContext glxContext;
   XVisualInfo * visualInfo;
   XSetWindowAttributes setableAttributes;

   /*
      Allocate hint structures.
   */
   if (NULL == (wm_hints = XAllocWMHints()) ||
      NULL == (class_hints = XAllocClassHint()) ||
      NULL == (size_hints = XAllocSizeHints())
   ) {
      fprintf(stderr,"%s: failure allocating memory.\n", XStuff.commandLineName);
      exit(1);
   }

   /*
      Create text properties.
   */
   if (0 == XStringListToTextProperty(&windowTitle, 1, &windowName) ||
      0 == XStringListToTextProperty(&iconTitle, 1, &iconName)
   ) {
      fprintf(stderr,"%s: structure allocation for text property failed.\n",
         XStuff.commandLineName);
      exit(1);
   }

   if (isOpenGLWindow) {
      int attributeList[] = {
         GLX_RGBA,
         GLX_DOUBLEBUFFER,
         GLX_RED_SIZE, 1,
         GLX_GREEN_SIZE, 1,
         GLX_BLUE_SIZE, 1,
         GLX_DEPTH_SIZE, 1,
         None
      };

      /* Open an X visual */
      if (! (visualInfo = glXChooseVisual(XStuff.display, XStuff.screen_num, attributeList))) {
         fprintf(stderr,"%s: can't open GL visual.\n", XStuff.commandLineName);
         exit(1);
      }

      /* Create a GLX context */
      if (!(glxContext = glXCreateContext(XStuff.display, visualInfo, None, GL_TRUE))) {
         fprintf(stderr,"%s: can't open GLX context.\n", XStuff.commandLineName);
         exit(1);
      }

      /* create a colormap */
      XStuff.colourMap = XCreateColormap(XStuff.display, XStuff.rootWindow, visualInfo->visual, AllocNone);
   }

   /*
      Create the main window.
   */
   if (isFullScreenAndBorderless) {
      windowX = 0;
      windowY = 0;
      XStuff.windowWidth = XStuff.screenWidth;
      XStuff.windowHeight = XStuff.screenHeight;
      windowBorderWidth = 0;
   }
   else {
      windowX = XStuff.screenWidth/3;
      windowY = XStuff.screenHeight/3;
      XStuff.windowWidth = XStuff.screenWidth/3;
      XStuff.windowHeight = XStuff.screenHeight/3;
      windowBorderWidth = 0;
   }

   if (isOpenGLWindow) {
      setableAttributes.colormap = XStuff.colourMap;
      setableAttributes.border_pixel = 0;

      XStuff.window = XCreateWindow(
         XStuff.display, XStuff.rootWindow,
         windowX, windowY, XStuff.windowWidth, XStuff.windowHeight,
         windowBorderWidth,
         visualInfo->depth, InputOutput, visualInfo->visual,
         CWBorderPixel | CWColormap,
         &setableAttributes);
   }
   else {
      XStuff.window = XCreateSimpleWindow(XStuff.display, XStuff.rootWindow,
         windowX, windowY, XStuff.windowWidth, XStuff.windowHeight,
         windowBorderWidth,
         XStuff.blackPixel, XStuff.whitePixel);
   }

   if (isFullScreenAndBorderless) {
      /*
         Note that this will prevent the user from being able
         to lower/raise/iconify/move/resize a window using the
         window manager.  Applications that do this should
         provide the user with hotkeys for lowering/raising/
         iconifying (see XLowerWindow(), XRaiseWindow(), XIconifyWindow() ).
         Also there seem to be some focus problems with doing
         what the below code does ... perhaps this could be fixed
         by calling one of XSetInputFocus(), XGetInputFocus(), XSetICFocus().
      */
      XSetWindowAttributes attributes;
      attributes.override_redirect = True;
      XChangeWindowAttributes(XStuff.display, XStuff.window, CWOverrideRedirect, &attributes);
   }

   /*
      Set window properties.
   */
   wm_hints->flags = StateHint | InputHint;
   wm_hints->initial_state = NormalState;   /* Should window be normal or iconified when first mapped ? */
   wm_hints->input = True;                  /* Does application need keyboard input? */
   if (None != iconPixmap) {
      wm_hints->flags |= IconPixmapHint;
      wm_hints->icon_pixmap = iconPixmap;
   }

   /* These are used by the window manager to get information */
   /* about this application from the resource database. */
   /*
      For example, with 4Dwm, you could add lines
      like these to your ~/.Xdefaults file :
              4Dwm*foo.clientDecoration: none
              4Dwm*goo.clientDecoration: border resizeh
      If foo is the name or class of an app's window, then that window will
      have no window decorations.  Similarly, in the case of goo, the window
      will only have a resizable border.  (The clientDecoration resource works
      the same way with Mwm.)
   */
   class_hints->res_name = XStuff.commandLineName;
   class_hints->res_class = "MyClassOfApplications";

   /* Use USPosition | USSize instead of PPosition | PSize to force hints. */
   /* size_hints->flags = PPosition | PSize | PMinSize; */
   size_hints->flags = PMinSize;
   /* In R4 and later, the x, y, width, and height members of XSizeHints should not be set. */
   size_hints->min_width = minWidth;
   size_hints->min_height = minHeight;
   if (isFullScreenAndBorderless)
      size_hints->flags |= (USPosition | USSize);

   XSetWMProperties(XStuff.display, XStuff.window, &windowName, &iconName, 
      argv, argc, size_hints, wm_hints,class_hints);

   /*
      Select event types wanted.

      ConfigureNotify events inform the app that the window has been
      resized.  Processing ConfigureNotify events is more efficient
      than having to call XGetGeometry() (which requires a reply from
      the server) on every Expose event.
   */
   XSelectInput(XStuff.display, XStuff.window,
      ExposureMask |
      KeyPressMask |
      KeyReleaseMask |
      ButtonPressMask |
      ButtonReleaseMask |
      PointerMotionMask |
      PointerMotionHintMask |
      StructureNotifyMask /* selects CirculateNotify, ConfigureNotify, DestroyNotify,
            GravityNotify, MapNotify, ReparentNotify, and UnmapNotify */
      );

   /*
      Display the window.
   */
   XMapWindow(XStuff.display, XStuff.window);

   if (isOpenGLWindow) {
      /* Connect the context to the window. */
      if (!glXMakeCurrent(XStuff.display, XStuff.window, glxContext)) {
         fprintf(stderr,"%s: can't set GL context.\n", XStuff.commandLineName);
         exit(1);
      }
   }

   if (!isOpenGLWindow) {
      /*
         Set the background color.
      */
      XSetWindowBackground(XStuff.display, XStuff.window, backgroundPixel);
   }

   // If you don't like these fonts, use xfontsel
   // to find the names of ones you like better.
   //
   char * nameOfIdealBigFont = "-adobe-helvetica-bold-o-normal-*-14-*";
   char * nameOfIdealSmallFont = "-sgi-screen-medium-*-*-*-15-*-*-*-*-*-*-*";

   // These fonts will be used if the
   // former ones cannot be found.
   //
   char * nameOfHighlyPortableBigFont = "9x15";
   char * nameOfHighlyPortableSmallFont = "fixed";

   if (NULL == (XStuff.bigFont = XLoadQueryFont(XStuff.display,nameOfIdealBigFont)))
      if (NULL == (XStuff.bigFont = XLoadQueryFont(XStuff.display,nameOfHighlyPortableBigFont)))
         fprintf(stderr,"%s: can't open font \"%s\"\nSome text may not appear.\n", 
            XStuff.commandLineName, nameOfHighlyPortableBigFont
         );

   if (NULL == (XStuff.smallFont = XLoadQueryFont(XStuff.display,nameOfIdealSmallFont)))
      if (NULL == (XStuff.smallFont = XLoadQueryFont(XStuff.display,nameOfHighlyPortableSmallFont)))
         fprintf(stderr,"%s: can't open font \"%s\"\nSome text may not appear.\n",
            XStuff.commandLineName, nameOfHighlyPortableSmallFont
         );

   /*
      Get a graphics context.
   */
   XStuff.gc = XCreateGC(XStuff.display, XStuff.window, 0, NULL);
   /* Specify black foreground since default window background */
   /* is white and default foreground is undefined. */
   XSetForeground(XStuff.display, XStuff.gc, XStuff.blackPixel);

   /*
      Get geometry information about window
   */
   if (False == XGetGeometry(XStuff.display, XStuff.window, &XStuff.rootWindow,
      &windowX, &windowY, &XStuff.windowWidth, &XStuff.windowHeight,
      &windowBorderWidth, &XStuff.screenDepth)
   ) {
      fprintf(stderr,"%s: can't get window geometry.\n", XStuff.commandLineName);
      exit(1);
   }
}

// ==================================================

// The below bitmaps were created with the command "bitmap [filename]".

#define cursor_width 19
#define cursor_height 19
#define cursor_x_hot 9
#define cursor_y_hot 9
static char cursor_bits[] = {
   0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00,
   0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x07, 0x00, 0x80, 0x08, 0x00,
   0x40, 0x10, 0x00, 0x7f, 0xf0, 0x07, 0x40, 0x10, 0x00, 0x80, 0x08, 0x00,
   0x00, 0x07, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00,
   0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00};
#define mask_width 19
#define mask_height 19
#define mask_x_hot 9
#define mask_y_hot 9
static char mask_bits[] = {
   0x00, 0x02, 0xf8, 0x00, 0x07, 0xf8, 0x00, 0x07, 0xf8, 0x00, 0x07, 0xf8,
   0x00, 0x07, 0xf8, 0x00, 0x07, 0xf8, 0x80, 0x0f, 0xf8, 0xc0, 0x1f, 0xf8,
   0xfe, 0xf8, 0xfb, 0xff, 0xf8, 0xff, 0xfe, 0xf8, 0xfb, 0xc0, 0x1f, 0xf8,
   0x80, 0x0f, 0xf8, 0x00, 0x07, 0xf8, 0x00, 0x07, 0xf8, 0x00, 0x07, 0xf8,
   0x00, 0x07, 0xf8, 0x00, 0x07, 0xf8, 0x00, 0x02, 0xf8};

void CreateCursorForWindow() {

   Pixmap source, mask;
   XColor foreground, background;

   foreground.pixel = 0;
   foreground.red = 0;
   foreground.green = 65535;
   foreground.blue = 0;
   foreground.flags = DoRed | DoGreen | DoBlue;
   foreground.pad = 0;

   background.pixel = 0;
   background.red = 0;
   background.green = 0;
   background.blue = 0;
   background.flags = DoRed | DoGreen | DoBlue;
   background.pad = 0;

   source = XCreateBitmapFromData(XStuff.display, XStuff.window,
                   cursor_bits, cursor_width, cursor_height);
   mask = XCreateBitmapFromData(XStuff.display, XStuff.window,
                   mask_bits, mask_width, mask_height);
   XStuff.cursor = XCreatePixmapCursor(XStuff.display, source, mask,
                   &foreground, &background,
                   cursor_x_hot, cursor_y_hot);

   XDefineCursor(XStuff.display, XStuff.window, XStuff.cursor);

   XFreePixmap(XStuff.display, source);
   XFreePixmap(XStuff.display, mask);
}

// ==================================================

// stuff for sound

#define STARTING_TRAIN_SOUND "chooChoo.au passingTrain.au"
#define DREADFUL_LAUGHTER_SOUND "laughter.au"
#define CRASH_SOUND "explosion.au carCrash.au"

void playSound( char * filename ) {

   char buffer[100];
#ifdef SGI
   sprintf( buffer, "sfplay %s &", filename );
#else
   sprintf( buffer, "audioplay -i %s &", filename );
#endif
   system(buffer);
}

// ==================================================

// global variables for the train track geometry
//
TrainTrack *track1, *track2, *track3, *track4, *track5, *track6;
float centerOfTrainTracksX, centerOfTrainTracksY, centerOfTrainTracksZ;
float radiusOfTrainTracks;
TrainTrackSwitch *switch1, *switch2, *switch3, *switch4;

// global variables for the trains
Train * train1, * train2;
Train::Sphere * geometryOfTrain1, * geometryOfTrain2;
bool * carCollisionsForTrain1, * carCollisionsForTrain2;

// global variables for the explosions
//
static const int MAX_NUMBER_OF_EXPLOSIONS = 30;
Explosion * arrayOfExplosions[ MAX_NUMBER_OF_EXPLOSIONS ];
int numberOfExplosions = 0;

void CreateTrainTracks() {
   const float S = 4.0;  // world scale factor
   const float ELEVATION_1 = 0.75; 
   const float ELEVATION_2 = 0.5;
   const float SWITCH_WIDTH = 0.2;
   const float SWITCH_LENGTH = 0.3;

   centerOfTrainTracksX = 0.75 * S;
   centerOfTrainTracksY = 0.75 * S;
   centerOfTrainTracksZ = 0.5 * ELEVATION_1 * S;
   radiusOfTrainTracks = 5.0 * S;

   track1 = new TrainTrack(
      -1.5 * S, 0.0, 0.0,
      1.0, 0.0, 0.0
   );
   track1->appendStraightSection( 4.5 * S, 0.0 );
   track2 = new TrainTrack(
      -1.5 * S, SWITCH_WIDTH * S, 0.0,
      1.0, 0.0, 0.0
   );
   track2->appendCurvedSection(  90.0, 0.5 * S, 0.5 * ELEVATION_1 * S );
   track2->appendCurvedSection( -90.0, 0.5 * S, 0.5 * ELEVATION_1 * S );
   track2->appendStraightSection( 2.5 * S, 0.0 );
   track2->appendCurvedSection( -90.0, 0.5 * S, - 0.5 * ELEVATION_1 * S );
   track2->appendCurvedSection(  90.0, 0.5 * S, - 0.5 * ELEVATION_1 * S );
   track3 = new TrainTrack(
      ( 3.0 + SWITCH_LENGTH )*S, 0.0, 0.0,
      1.0, 0.0, 0.0
   );
   track3->appendCurvedSection( -90.0, S, 0.0 );
   track3->appendStraightSection( S, 0.0 );
   track3->appendCurvedSection( -90.0, S, - ELEVATION_2 * S );
   track3->appendStraightSection( ( 4.5 + 2*SWITCH_LENGTH )*S, 0.0 );
   track3->appendCurvedSection( -90.0, S, 0.0 );
   track3->appendStraightSection( S, 0.0 );
   track3->appendCurvedSection( -90.0, S, ELEVATION_2 * S );

   track4 = new TrainTrack(
      0.0, -1.5 * S, 0.0,
      0.0, 1.0, 0.0
   );
   track4->appendStraightSection( 4.5 * S, 0.0 );
   track5 = new TrainTrack(
      SWITCH_WIDTH * S, -1.5 * S, 0.0,
      0.0, 1.0, 0.0
   );
   track5->appendCurvedSection( -90.0, 0.5 * S, 0.5 * ELEVATION_1 * S );
   track5->appendCurvedSection(  90.0, 0.5 * S, 0.5 * ELEVATION_1 * S );
   track5->appendStraightSection( 2.5 * S, 0.0 );
   track5->appendCurvedSection(  90.0, 0.5 * S, - 0.5 * ELEVATION_1 * S );
   track5->appendCurvedSection( -90.0, 0.5 * S, - 0.5 * ELEVATION_1 * S );
   track6 = new TrainTrack(
      0.0, ( 3.0 + SWITCH_LENGTH )*S, 0.0,
      0.0, 1.0, 0.0
   );
   track6->appendCurvedSection(  90.0, S, 0.0 );
   track6->appendStraightSection( S, 0.0 );
   track6->appendCurvedSection(  90.0, S, ELEVATION_2 * S );
   track6->appendStraightSection( ( 4.5 + 2*SWITCH_LENGTH )*S, 0.0 );
   track6->appendCurvedSection(  90.0, S, 0.0 );
   track6->appendStraightSection( S, 0.0 );
   track6->appendCurvedSection(  90.0, S, - ELEVATION_2 * S );


   switch1 = new TrainTrackSwitch(
      track1, AbstractTrainTrack::INITIAL_END_POINT,
      track3, AbstractTrainTrack::FINAL_END_POINT,
      track2, AbstractTrainTrack::INITIAL_END_POINT,
      track3, AbstractTrainTrack::FINAL_END_POINT
   );
   switch2 = new TrainTrackSwitch(
      track1, AbstractTrainTrack::FINAL_END_POINT,
      track3, AbstractTrainTrack::INITIAL_END_POINT,
      track2, AbstractTrainTrack::FINAL_END_POINT,
      track3, AbstractTrainTrack::INITIAL_END_POINT
   );
   switch3 = new TrainTrackSwitch(
      track4, AbstractTrainTrack::INITIAL_END_POINT,
      track6, AbstractTrainTrack::FINAL_END_POINT,
      track5, AbstractTrainTrack::INITIAL_END_POINT,
      track6, AbstractTrainTrack::FINAL_END_POINT
   );
   switch4 = new TrainTrackSwitch(
      track4, AbstractTrainTrack::FINAL_END_POINT,
      track6, AbstractTrainTrack::INITIAL_END_POINT,
      track5, AbstractTrainTrack::FINAL_END_POINT,
      track6, AbstractTrainTrack::INITIAL_END_POINT
   );
   switch3->toggle();
   switch4->toggle();

   train1 = new Train( track3, 23.0, TrainCar::BACKWARD_DIRECTION );
   train1->appendCar( TrainCar::LOCOMOTIVE );
   train1->appendCar( TrainCar::FREIGHT_CAR );
   train1->appendCar( TrainCar::FREIGHT_CAR );
   train1->appendCar( TrainCar::FREIGHT_CAR );
   train1->appendCar( TrainCar::FREIGHT_CAR );
   train1->appendCar( TrainCar::CABOOSE );

   train2 = new Train( track6, 16.0, TrainCar::BACKWARD_DIRECTION );
   train2->appendCar( TrainCar::LOCOMOTIVE, 0.3, 0.3, 0.4 );
   train2->appendCar( TrainCar::FREIGHT_CAR, 0.9, 0.9, 0.2 );
   train2->appendCar( TrainCar::FREIGHT_CAR, 0.9, 0.9, 0.2 );
   train2->appendCar( TrainCar::FREIGHT_CAR, 0.9, 0.9, 0.2 );
   train2->appendCar( TrainCar::FREIGHT_CAR, 0.9, 0.9, 0.2 );
   train2->appendCar( TrainCar::CABOOSE );

   geometryOfTrain1 = new Train::Sphere [ train1->getLength() ];
   geometryOfTrain2 = new Train::Sphere [ train2->getLength() ];
   carCollisionsForTrain1 = new bool [ train1->getLength() ];
   carCollisionsForTrain2 = new bool [ train2->getLength() ];

   for ( int j = 0; j < MAX_NUMBER_OF_EXPLOSIONS; ++j )
      arrayOfExplosions[ j ] = NULL;
   numberOfExplosions = 0;
}

void DestroyTrainTracks() {

   delete track1; track1 = NULL;
   delete track2; track2 = NULL;
   delete track3; track3 = NULL;
   delete track4; track4 = NULL;
   delete track5; track5 = NULL;
   delete track6; track6 = NULL;

   delete switch1; switch1 = NULL;
   delete switch2; switch2 = NULL;
   delete switch3; switch3 = NULL;
   delete switch4; switch4 = NULL;

   delete train1; train1 = NULL;
   delete train2; train2 = NULL;

   delete [] geometryOfTrain1; geometryOfTrain1 = NULL;
   delete [] geometryOfTrain2; geometryOfTrain2 = NULL;
   delete [] carCollisionsForTrain1; carCollisionsForTrain1 = NULL;
   delete [] carCollisionsForTrain2; carCollisionsForTrain2 = NULL;
}

OrbitingCamera * pOCamera = NULL;
FreeFloatingCamera * pFFCamera1 = NULL;
FreeFloatingCamera * pFFCamera2 = NULL;
FreeFloatingCamera * pCurrentFFCamera = NULL;
bool useOrbitingCamera = true;

// ==================================================

void Refresh() {

   glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);   // clear image & z-buffer

   if ( useOrbitingCamera ) {
      pOCamera->invokeTransform();
   }
   else {
      const TrainCar * trainCar
         = ( pCurrentFFCamera == pFFCamera1 ? train1 : train2 )->getCar( 0 );
      float x, y, z;
      trainCar->getPosition( x, y, z );
      float d_x, d_y, d_z;
      trainCar->getDirection( d_x, d_y, d_z );
      pCurrentFFCamera->invokeTransformWithOffset(
         x, y, z, d_x, d_y, d_z
      );
   }

   glMatrixMode(GL_MODELVIEW);   // set transformation mode to model/view transformations
   glLoadIdentity();             // clear matrix

   track1->draw( LOD, isWireframe );
   track2->draw( LOD, isWireframe );
   track3->draw( LOD, isWireframe );
   track4->draw( LOD, isWireframe );
   track5->draw( LOD, isWireframe );
   track6->draw( LOD, isWireframe );
   switch1->draw( LOD, isWireframe );
   switch2->draw( LOD, isWireframe );
   switch3->draw( LOD, isWireframe );
   switch4->draw( LOD, isWireframe );
   train1->draw( LOD, isWireframe );
   train2->draw( LOD, isWireframe );
   for ( int j = 0; j < numberOfExplosions; ++j )
      arrayOfExplosions[ j ]->draw( LOD, isWireframe );

   glFlush();
   glXSwapBuffers(XStuff.display, XStuff.window);

   if (bDisplayText) {

      // draw the menu
      //
      if (NULL != theMenu)
         theMenu->draw( XStuff.gc, currentTextColour, XStuff.blackPixel );

      // draw strings
      //
      char buffer[100];
      int x_offset = 5, y_offset = MENU_HEIGHT, charHeight;

      if (NULL != XStuff.smallFont) {
         XSetFont(XStuff.display, XStuff.gc, XStuff.smallFont->fid);
         charHeight = (int)( 1.2
            * ( XStuff.smallFont->ascent + XStuff.smallFont->descent )
         );

         if ( NULL != currentHelpString ) {
            strcpy( buffer, currentHelpString );
            XDrawString(
               XStuff.display, XStuff.window, XStuff.gc,
               x_offset, y_offset + charHeight, buffer, strlen(buffer)
            );
            y_offset += charHeight;
         }
      }

#if 0
      if (NULL != XStuff.bigFont) {
         XSetFont(XStuff.display, XStuff.gc, XStuff.bigFont->fid);
         charHeight = (int)( 1.2
            * ( XStuff.bigFont->ascent + XStuff.bigFont->descent )
         );

         sprintf(buffer,"blah" );
         XDrawString(
            XStuff.display, XStuff.window, XStuff.gc,
            x_offset, y_offset + charHeight, buffer, strlen(buffer)
         );
         y_offset += charHeight;
         sprintf(buffer,"ghah" );
         XDrawString(
            XStuff.display, XStuff.window, XStuff.gc,
            x_offset, y_offset + charHeight, buffer, strlen(buffer)
         );
      }
#endif

      // If the user hits the key to advance time many times in rapid
      // succession, the cpu will start doing some heavy computations
      // to try and keep up, displaying frame after frame as time
      // advances.  Meanwhile, requests to the X server
      // to draw strings can queue up, and finally all get dumped
      // onto the last frame.  The user sees this as text
      // overwriting itself, becoming illegible.  By calling
      // XFlush(), we prevent the requests from queuing up.
      //
      XFlush(XStuff.display);
   }
}

// ==================================================

const char * determineHelpString( int mouse_x, int mouse_y ) {

   const char * string = theMenu->getDescription( mouse_x, mouse_y );
   if ( NULL == string ) {
      string = useOrbitingCamera
         ? orbitingCameraHelpString
         : freeFloatingCameraHelpString;
   }
   return string;
}

void HandleKeyPress( int keyCode, const XEvent & event, bool & userHasQuit ) {

   // Consult /usr/include/X11/keysymdef.h for keysym codes
   //
   switch ( keyCode ) {
      case XK_Home:
         break;
      case XK_End:
         break;
      case XK_Page_Up:
         break;
      case XK_Page_Down:
         break;

      case XK_Up:
         delta_t *= 1.5;
         break;
      case XK_Down:
         delta_t /= 1.5;
         break;
      case XK_space:
         isSimulationPaused = ! isSimulationPaused;
         break;

      case XK_F6:
         // toggle back face culling
         bBackFaceCulling = !bBackFaceCulling;
         if (bBackFaceCulling) {
            glEnable(GL_CULL_FACE);
            glCullFace(bFrontFacing ? GL_BACK : GL_FRONT);
         }
         else
            glDisable(GL_CULL_FACE);
         Refresh();
         break;
      case XK_F7:
         bFrontFacing = !bFrontFacing;
         glCullFace(bFrontFacing ? GL_BACK : GL_FRONT);
         Refresh();
         break;
      case XK_F8:
         bShadingIsSmooth = !bShadingIsSmooth;
         glShadeModel(bShadingIsSmooth ? GL_SMOOTH : GL_FLAT);
         Refresh();
         break;
      case XK_F9:
         bDisplayText = ! bDisplayText;
         Refresh();
         break;
#ifdef TOGGLEABLE_WINDOW_DECORATIONS
      case XK_F10:
         PropMotifWmHints mwm_hints;
         Atom mwm_hints_atom;

         bNoWindowDecorations = ! bNoWindowDecorations;
         mwm_hints_atom  = XInternAtom(
            XStuff.display, _XA_MOTIF_WM_HINTS, False
         );
         mwm_hints.flags = (int) MWM_HINTS_DECORATIONS;
         mwm_hints.decorations = bNoWindowDecorations ? 0
            : MWM_DECOR_ALL;
            //: MWM_DECOR_BORDER | MWM_DECOR_TITLE | MWM_DECOR_RESIZEH
            //| MWM_DECOR_MINIMIZE | MWM_DECOR_MAXIMIZE | MWM_DECOR_MENU;
         XChangeProperty (
            XStuff.display, XStuff.window, mwm_hints_atom,
            mwm_hints_atom,
            32, PropModeReplace, (unsigned char *) & mwm_hints,
            PROP_MWM_HINTS_ELEMENTS
         );
         break;
#endif
      case XK_e:
      case XK_E:
         if ( train1->isEngineOn() || train2->isEngineOn() ) {
            train1->stop();
            train2->stop();
         }
         else {
            train1->start();
            train2->start();
            isTheSceneAnimating = true;
            playSound( STARTING_TRAIN_SOUND );
         }
         break;
      case XK_f:
      case XK_F:
         simulateSoftwareFailure = true;
         playSound( DREADFUL_LAUGHTER_SOUND );
         break;
      case XK_r:
      case XK_R:
         train1->reset();
         train2->reset();
         switch1->reset();
         switch2->reset();
         switch3->reset();
         switch3->toggle();
         switch4->reset();
         switch4->toggle();
         isTheSceneAnimating = false;
         simulateSoftwareFailure = false;
         for ( int j = 0; j < numberOfExplosions; ++j ) {
            delete arrayOfExplosions[ j ];
            arrayOfExplosions[ j ] = NULL;
         }
         numberOfExplosions = 0;
         Refresh();
         break;
      case XK_c:
      case XK_C:
         if ( useOrbitingCamera ) {
            useOrbitingCamera = false;
            pCurrentFFCamera = pFFCamera1;
         }
         else if ( pCurrentFFCamera == pFFCamera1 ) {
            pCurrentFFCamera = pFFCamera2;
         }
         else {
            useOrbitingCamera = true;
         }

         currentHelpString = determineHelpString( event.xkey.x, event.xkey.y );

         Refresh();
         break;
      case XK_l:
      case XK_L:
         switch ( LOD ) {
         case    LOW_LOD : LOD = MEDIUM_LOD; break;
         case MEDIUM_LOD : LOD =   HIGH_LOD; break;
         case   HIGH_LOD : LOD =    LOW_LOD; break;
         }
         Refresh();
         break;
      case XK_w:
      case XK_W:
         isWireframe = ! isWireframe;
         Refresh();
         break;
      case XK_1:
      case XK_2:
      case XK_3:
      case XK_4:
      case XK_5:
      case XK_6:
      case XK_7:
      case XK_8:
         // change colour of something
         int index;
         GLfloat rgb[4];
         index = keyCode - XK_1;
         rgb[0] = (GLfloat)(index & 1);
         rgb[1] = (GLfloat)((index >> 1) & 1);
         rgb[2] = (GLfloat)((index >> 2) & 1);
         rgb[3] = 1.0;

         if (event.xkey.state & ShiftMask) {
            // Shift
            glClearColor(0.75*rgb[0], 0.75*rgb[1], 0.75*rgb[2], 0.0);
         }
         else {
            // no modifiers
            FreePixelColour(currentTextColour);
            currentTextColour = AllocPixelColour(
               0.75*rgb[0], 0.75*rgb[ 1], 0.75*rgb[2]
            );
            XSetForeground(XStuff.display, XStuff.gc, currentTextColour);
         }
         Refresh();
         break;
      case XK_Escape:
         // time to exit
         userHasQuit = true;
         break;
   }
}

// ==================================================

void MainLoop() {

   struct {
      bool LMB, MMB, RMB;
      bool Ctrl, Alt, Shift;
      bool isInteractingWithMenu;
   } oldMouseState = {
      false, false, false,
      false, false, false,
      false
   }, newMouseState;
   int oldMouseX, oldMouseY, newMouseX, newMouseY;

   Window dummyWindow1, dummyWindow2;
   XEvent event;
   bool userHasQuit = false;
   int dummyInt1, dummyInt2,
       keyCode;
   unsigned int StateOfModifiers;
   
   while ( ! userHasQuit ) {
      if ( isTheSceneAnimating && ! isSimulationPaused ) {
         while (0 == XEventsQueued(XStuff.display, QueuedAfterFlush)) {
            // Perform animation.
            //
            if (isTheSceneAnimating) {

               // animate the scene

               bool isLapComplete;

               bool isTrain1Moving = train1->advance( delta_t, isLapComplete );
               if ( isLapComplete ) {
                  switch1->toggle();
                  switch2->toggle();
               }

               bool isTrain2Moving = train2->advance( delta_t, isLapComplete );
               if ( isLapComplete && ! simulateSoftwareFailure ) {
                  switch3->toggle();
                  switch4->toggle();
               }

               bool areFiresBurning = false;
               int j;
               for ( j = 0; j < numberOfExplosions; ++j ) {
                  arrayOfExplosions[ j ]->burn( delta_t );
                  if ( arrayOfExplosions[ j ]->isStillBurning() ) {
                     areFiresBurning = true;
                  }
               }
               // Clean up explosions that have finished burning out.
               // This is kinda hackish, since we only clean up
               // starting at the end of the array and working backwards
               // until we hit an explosion that is still burning.
               // It would be much nicer if we had a linked list of explosions.
               for ( j = numberOfExplosions-1; j >= 0; --j ) {
                  if ( ! arrayOfExplosions[ j ]->isStillBurning() ) {
                     delete arrayOfExplosions[ j ];
                     arrayOfExplosions[ j ] = NULL;
                     -- numberOfExplosions;
                  }
                  else break;
               }

               isTheSceneAnimating
                  = isTrain1Moving || isTrain2Moving || areFiresBurning;

               if ( isTheSceneAnimating ) {
                  train1->getBoundingGeometry( geometryOfTrain1 );
                  train2->getBoundingGeometry( geometryOfTrain2 );
                  Train::performCollisionTest(
                     train1->getLength(), geometryOfTrain1,
                     train2->getLength(), geometryOfTrain2,
                     carCollisionsForTrain1, carCollisionsForTrain2
                  );
                  bool crashOccured = false;
                  for ( j = 0; j < train1->getLength(); ++j ) {
                     if ( carCollisionsForTrain1[ j ] ) {
                        crashOccured = true;
                        train1->removeCar( j );
                        if ( numberOfExplosions < MAX_NUMBER_OF_EXPLOSIONS ) {
                           arrayOfExplosions[ numberOfExplosions ]
                              = new Explosion( train1->getCar( j ) );
                           ++ numberOfExplosions;
                        }
                     }
                  }
                  for ( j = 0; j < train2->getLength(); ++j ) {
                     if ( carCollisionsForTrain2[ j ] ) {
                        crashOccured = true;
                        train2->removeCar( j );
                        if ( numberOfExplosions < MAX_NUMBER_OF_EXPLOSIONS ) {
                           arrayOfExplosions[ numberOfExplosions ]
                              = new Explosion( train2->getCar( j ) );
                           ++ numberOfExplosions;
                        }
                     }
                  }
                  if (crashOccured) {
                     playSound( CRASH_SOUND );
                  }
               }
            }
            Refresh();
         }
      }

      XNextEvent(XStuff.display, &event);
      switch (event.type) {
         case Expose:
            // Unless this is the last contiguous expose, don't redraw.
            if (event.xexpose.count != 0)
               break;
            // Only *contiguous* expose events can be skipped.  Searching through the event queue
            // to find the last expose event would skip over intervening ConfigureNotify events.

            Refresh();
            break;
         case ConfigureNotify:
            // window has been resized
            if (
               XStuff.windowWidth != event.xconfigure.width ||
               XStuff.windowHeight != event.xconfigure.height
            ) {
               XStuff.windowWidth = event.xconfigure.width;
               XStuff.windowHeight = event.xconfigure.height;

               if (NULL != theMenu)
                  theMenu->resize( 0, 0, XStuff.windowWidth, MENU_HEIGHT );

               pOCamera->resizeViewport(
                  XStuff.windowWidth, XStuff.windowHeight,
                  0, MENU_HEIGHT, XStuff.windowWidth, XStuff.windowHeight
               );
               pFFCamera1->resizeViewport(
                  XStuff.windowWidth, XStuff.windowHeight,
                  0, MENU_HEIGHT, XStuff.windowWidth, XStuff.windowHeight
               );
               pFFCamera2->resizeViewport(
                  XStuff.windowWidth, XStuff.windowHeight,
                  0, MENU_HEIGHT, XStuff.windowWidth, XStuff.windowHeight
               );
               Refresh();
            }
            break;
         case KeyPress:
            keyCode = XLookupKeysym(&event.xkey, 0);
            HandleKeyPress( keyCode, event, userHasQuit );
            break;
         case KeyRelease:
            break;
         case ButtonPress:
            oldMouseState.LMB   = (event.xbutton.state & Button1Mask) != 0;
            oldMouseState.MMB   = (event.xbutton.state & Button2Mask) != 0;
            oldMouseState.RMB   = (event.xbutton.state & Button3Mask) != 0;
            oldMouseState.Ctrl  = (event.xbutton.state & ControlMask) != 0;
            oldMouseState.Alt   = (event.xbutton.state & Mod1Mask) != 0;
            oldMouseState.Shift = (event.xbutton.state & ShiftMask) != 0;
            oldMouseX = event.xbutton.x;
            oldMouseY = event.xbutton.y;

            if ( event.xbutton.y <= MENU_HEIGHT ) {
               theMenu->buttonDown( event.xbutton.x, event.xbutton.y );
               oldMouseState.isInteractingWithMenu = true;
               Refresh();
            }
            break;
         case ButtonRelease:
            if ( oldMouseState.isInteractingWithMenu ) {
               keyCode = theMenu->buttonUp( event.xbutton.x, event.xbutton.y );
               if ( Menu::NO_CLIENT_DATA != keyCode ) {
                  HandleKeyPress( keyCode, event, userHasQuit );
               }
               oldMouseState.isInteractingWithMenu = false;
            }

            oldMouseState.LMB   = (event.xbutton.state & Button1Mask) != 0;
            oldMouseState.MMB   = (event.xbutton.state & Button2Mask) != 0;
            oldMouseState.RMB   = (event.xbutton.state & Button3Mask) != 0;
            oldMouseState.Ctrl  = (event.xbutton.state & ControlMask) != 0;
            oldMouseState.Alt   = (event.xbutton.state & Mod1Mask) != 0;
            oldMouseState.Shift = (event.xbutton.state & ShiftMask) != 0;
            oldMouseX = event.xbutton.x;
            oldMouseY = event.xbutton.y;
            break;
         case MotionNotify:
            if ( oldMouseState.isInteractingWithMenu )
               break;

            // if any of the buttons are down ...
            if ( event.xmotion.state
               & ( Button1Mask | Button2Mask | Button3Mask )
            ) {
               XQueryPointer(XStuff.display, XStuff.window, &dummyWindow1,
                  &dummyWindow2, &dummyInt1, &dummyInt2,
                  &newMouseX, &newMouseY, &StateOfModifiers
               );
               newMouseState.LMB   = (StateOfModifiers & Button1Mask) != 0;
               newMouseState.MMB   = (StateOfModifiers & Button2Mask) != 0;
               newMouseState.RMB   = (StateOfModifiers & Button3Mask) != 0;
               newMouseState.Ctrl  = (StateOfModifiers & ControlMask) != 0;
               newMouseState.Alt   = (StateOfModifiers & Mod1Mask) != 0;
               newMouseState.Shift = (StateOfModifiers & ShiftMask) != 0;
               newMouseState.isInteractingWithMenu
                  = oldMouseState.isInteractingWithMenu;

               if (
                  memcmp(
                     (void*)&newMouseState,
                     (void*)&oldMouseState,
                     sizeof( newMouseState )
                  ) == 0
               ) {
                  float delta_x_right = newMouseX - oldMouseX;
                  float delta_y_up = oldMouseY - newMouseY;

                  if ( ! newMouseState.Ctrl ) {
                     if ( newMouseState.LMB ) {
                        if (useOrbitingCamera) {
                           pOCamera->orbit(
                              oldMouseX, oldMouseY, newMouseX, newMouseY
                           );
                        }
                        else {
                           pCurrentFFCamera->yawCameraRight( delta_x_right );
                           pCurrentFFCamera->pitchCameraUp( delta_y_up );
                        }
                        Refresh();
                     }
                     else if ( newMouseState.MMB ) {
                        if (useOrbitingCamera) {
                           pOCamera->translateSceneRightAndUp(
                              delta_x_right, delta_y_up
                           );
                        }
                        else {
                           pCurrentFFCamera->translateCameraRightAndUp(
                              delta_x_right, delta_y_up
                           );
                        }
                        Refresh();
                     }
                     else if ( newMouseState.RMB ) {
                        if (useOrbitingCamera) {
                           pOCamera->zoomIn( delta_y_up );
                        }
                        else {
                           pCurrentFFCamera->dollyCameraForward( delta_y_up );
                        }
                        Refresh();
                     }
                  }
                  else {
                     if ( newMouseState.LMB ) {
                        if ( ! useOrbitingCamera ) {
                           pCurrentFFCamera->rollCameraRight( delta_x_right );
                        }
                        Refresh();
                     }
                     else if ( newMouseState.RMB ) {
                        if (useOrbitingCamera) {
                           pOCamera->dollyCameraForward( delta_y_up );
                        }
                        else {
                           pCurrentFFCamera->zoomIn( delta_y_up );
                        }
                        Refresh();
                     }
                  }
               }

               oldMouseState = newMouseState;
               oldMouseX = newMouseX;
               oldMouseY = newMouseY;
            }
            else { // none of the mouse buttons are down

               // This is rather inneficient over an X connection, but
               // it's the only way I know of to make rollover help work.
               //
               XQueryPointer(XStuff.display, XStuff.window, &dummyWindow1,
                  &dummyWindow2, &dummyInt1, &dummyInt2,
                  &newMouseX, &newMouseY, &StateOfModifiers
               );

               const char * newHelpString
                  = determineHelpString( newMouseX, newMouseY );
               if ( newHelpString != currentHelpString ) {
                  currentHelpString = newHelpString;
                  Refresh();
               }
            }
            break;
      }   /* switch(event */
   }   /* while ( ! userHasQuit ) */
}

// ==================================================

void main(int argc, char *argv[]) {

   char * display_name = NULL;       /* Server to connect to */

   XStuff.commandLineName = basename(argv[0]);

   // Connect to the X server.
   //
   if (NULL == (XStuff.display = XOpenDisplay(display_name))) {
      fprintf(stderr,"%s: can't connect to X server %s\n",
         XStuff.commandLineName, XDisplayName(display_name));
      exit(1);
   }
   XStuff.screen_num     = DefaultScreen         (XStuff.display);
   XStuff.screen_ptr     = DefaultScreenOfDisplay(XStuff.display);
   XStuff.screenWidth    = DisplayWidth          (XStuff.display, XStuff.screen_num);
   XStuff.screenHeight   = DisplayHeight         (XStuff.display, XStuff.screen_num);
   XStuff.screenWidthMM  = DisplayWidthMM        (XStuff.display, XStuff.screen_num);
   XStuff.screenHeightMM = DisplayHeightMM       (XStuff.display, XStuff.screen_num);
   XStuff.rootWindow     = RootWindow            (XStuff.display, XStuff.screen_num);
   XStuff.blackPixel     = BlackPixel            (XStuff.display, XStuff.screen_num);
   XStuff.whitePixel     = WhitePixel            (XStuff.display, XStuff.screen_num);

   printf(
      "This program simulates a toy train set.\n"
      "The user interacts with the program using the mouse, menu buttons,\n"
      "and (optionally) the following keys:\n"
      "  \"e\"                Toggle engines\n"
      "  \"f\"                Simulate switching-software failure\n"
      "  \"r\"                Reset trains\n"
      "  \"c\"                Cycle through camera views\n"
      "  \"l\"                Cycle through different levels-of-detail\n"
      "  \"w\"                Toggle wireframe geometry\n"
      "  Escape             Quit\n"
      "  up/down arrows     Increase/decrease simulation speed\n"
      "  space bar          Pause simulation\n"

      // not yet implemented
      //"  F4                 Reset current camera view\n"

      "  F6                 Toggle backface culling\n"
      "  F7                 Toggle which faces are front facing\n"
      "  F8                 Toggle smooth/flat shading\n"
      "  F9                 Toggle text and menu display\n"
#ifdef TOGGLEABLE_WINDOW_DECORATIONS
      "  F10                Toggle window decorations\n"
      "                     (requires Motif-like window manager)\n"
#endif
      "  \"1\"-\"8\"            Select text & menu colour\n"
      "  Shift-\"1\"-\"8\"      Select background colour\n"
      "Press Enter to begin ... "
   );
   getchar();

   MakeWindow("Imprudent Trainlines Inc.","Train",None,0,0,False,True,XStuff.blackPixel,argc,argv);
   CreateCursorForWindow();

   // this will (hopefully) bring up a program for controlling the speaker volume
   //
#ifdef SGI
   system("apanel &");
#else
   system("audiocontrol &");
#endif

   // Initialize some X stuff.
   //
   currentTextColour = AllocPixelColour(0.0, 0.75, 0.75);
   XSetForeground(XStuff.display, XStuff.gc, currentTextColour);

   // Create a menu.
   //
   if (NULL != XStuff.smallFont) {
      theMenu = new Menu(
         XStuff.display, XStuff.window, XStuff.smallFont,
         0, 0, XStuff.windowWidth - 1, MENU_HEIGHT - 1
      );
      theMenu->appendMenuItem( "Engines", "Start / stop train engines", XK_e );
      theMenu->appendMenuItem( "Fail", "Simulate software failure", XK_f );
      theMenu->appendMenuItem(
         "Reset", "Reset trains to initial position", XK_r
      );
      theMenu->appendMenuItem(
         "Camera 1/2/3", "Cycle through different camera views", XK_c
      );
      theMenu->appendMenuItem(
         "LevelOfDetail", "Cycle through low to high levels of detail", XK_l
      );
      theMenu->appendMenuItem(
         "Wireframe", "Toggle wireframe geometry", XK_w
      );
      theMenu->appendMenuItem(
         "Exit", "Terminate application", XK_Escape
      );
   }

   // Initialize some OpenGL stuff.

   if (bBackFaceCulling) {
      // turn on back face culling
      glEnable(GL_CULL_FACE);
      glCullFace(bFrontFacing ? GL_BACK : GL_FRONT);
   }

   // Don't bother doing this, since the normals are
   // already normalized by us at generation time.
   // There's no need to ask OpenGL to explicitly try
   // to normalize again and again every time the
   // user rotates the view.
   //
   // glEnable(GL_NORMALIZE);   // This asks that normal vectors be normalized for us, automatically

   glClearColor(0.0, 0.0, 0.0, 0.0);   // this sets the colour to use when glClear() is called
   glClearDepth(1.0);
   // glDepthFunc(GL_LEQUAL);
   glEnable(GL_DEPTH_TEST);

   // Don't bother doing this
   //
   // // Clear the buffers.
   // //
   // glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
   // glXSwapBuffers(display, window);
   // glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
   // glXSwapBuffers(display, window);

   // Setup colours & lighting properties of the materiel.
   //
   GLfloat specular[] = { 1.0, 1.0, 1.0, 1.0 };
   GLfloat colourFront[] = { 0.0, 1.0, 1.0, 1.0 };   // colour of front faces of object
   GLfloat colourBack[] = { 0.0, 1.0, 0.0, 1.0 };    // colour of back faces of object
   GLfloat shininess[] = { 50.0 };
   glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, specular);
   glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, colourFront);
   glMaterialfv(GL_BACK, GL_AMBIENT_AND_DIFFUSE, colourBack);
   glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, shininess);

   // Setup lighting.
   //
   GLfloat lightPosition[] = { 10.0, 10.0, 10.0, 0.0 }; 
   glLightfv(GL_LIGHT0, GL_POSITION, lightPosition);
   glEnable(GL_LIGHTING);
   glEnable(GL_LIGHT0);
   GLfloat ambientLightColour[] = { 0.5, 0.5, 0.5, 1.0 };
   glLightfv(GL_LIGHT1, GL_AMBIENT, ambientLightColour );
   glEnable(GL_LIGHT1);
   GLfloat lightModelFlag[1] = { 1.0 };
   glLightModelfv(GL_LIGHT_MODEL_TWO_SIDE, lightModelFlag);


   // Build up a scene
   //
   CreateTrainTracks();

   // Create cameras
   //
   pOCamera = new OrbitingCamera(
      XStuff.windowWidth, XStuff.windowHeight,
      0, MENU_HEIGHT, XStuff.windowWidth, XStuff.windowHeight,
      centerOfTrainTracksX, centerOfTrainTracksY, centerOfTrainTracksZ,
      radiusOfTrainTracks
   );
   pOCamera->orbit(
      0.5 * XStuff.windowWidth, 0.5 * XStuff.windowHeight,
      0.67 * XStuff.windowWidth, 0.67 * XStuff.windowHeight
   );
   pFFCamera1 = new FreeFloatingCamera(
      XStuff.windowWidth, XStuff.windowHeight,
      0, MENU_HEIGHT, XStuff.windowWidth, XStuff.windowHeight,
      0.05,       // viewport radius
      0.1, 500.0, // near/far planes
      2.5, 100.0  // translational/rotational speed
   );
   pFFCamera1->translateCameraRightAndUp( 400, 150 );
   pFFCamera1->dollyCameraForward( 200 );
   pFFCamera1->yawCameraRight( -185 );
   pFFCamera1->pitchCameraUp( -10 );
   pFFCamera2 = new FreeFloatingCamera(
      XStuff.windowWidth, XStuff.windowHeight,
      0, MENU_HEIGHT, XStuff.windowWidth, XStuff.windowHeight,
      0.05,       // viewport radius
      0.1, 500.0, // near/far planes
      2.5, 100.0  // translational/rotational speed
   );
   pFFCamera2->translateCameraRightAndUp( 0, 500 );
   pFFCamera2->dollyCameraForward( -600 );
   pFFCamera2->pitchCameraUp( -55 );

   // Loop for events until user quits.
   //
   MainLoop();

   // Clean up.
   //
   DestroyTrainTracks();
   delete theMenu;
   FreePixelColour(currentTextColour);
   if (NULL != XStuff.bigFont)
      XUnloadFont(XStuff.display, XStuff.bigFont->fid);
   if (NULL != XStuff.smallFont)
      XUnloadFont(XStuff.display, XStuff.smallFont->fid);
   XFreeGC(XStuff.display, XStuff.gc);
   XFreeCursor(XStuff.display, XStuff.cursor);
   XCloseDisplay(XStuff.display);
}

