
#include "Camera.h"
#include <GL/gl.h>


Camera::Camera(
   int window_width, int window_height,
   int viewport_x1, int viewport_y1,
   int viewport_x2, int viewport_y2,
   float viewport_radius,
   float near_plane, float far_plane
) :
   _initial_viewport_radius( viewport_radius ),
   _initial_near_plane( near_plane ), _initial_far_plane( far_plane ),
   _viewport_radius( _initial_viewport_radius ),
   _near_plane( _initial_near_plane ), _far_plane( _initial_far_plane )
{
   ASSERT( viewport_x1 < viewport_x2 );
   ASSERT( viewport_y1 < viewport_y2 );
   resizeViewport(
      window_width, window_height,
      viewport_x1, viewport_y1, viewport_x2, viewport_y2
   );
}

void Camera::reset() {

   _viewport_radius = _initial_viewport_radius;
   _near_plane = _initial_near_plane;
   _far_plane = _initial_far_plane;
}

void Camera::invokeTransform() {

   glMatrixMode( GL_PROJECTION );
   glLoadIdentity();  // clear matrix

   // for orthographic projection, use glOrtho() instead of glFrustum()
   //
   glFrustum(
      - 0.5 * _viewport_width,  0.5 * _viewport_width,    // left, right
      - 0.5 * _viewport_height, 0.5 * _viewport_height,   // bottom, top
      _near_plane, _far_plane
   );
}

void Camera::resizeViewport(
   int window_width, int window_height,
   int viewport_x1, int viewport_y1,
   int viewport_x2, int viewport_y2
) {
   ASSERT( viewport_x1 < viewport_x2 );
   ASSERT( viewport_y1 < viewport_y2 );

   _window_width = window_width;
   _window_height = window_height;
   _viewport_x1 = viewport_x1;
   _viewport_y1 = viewport_y1;
   _viewport_x2 = viewport_x2;
   _viewport_y2 = viewport_y2;
   _viewport_cx = ( _viewport_x1 + _viewport_x2 ) / 2;
   _viewport_cy = ( _viewport_y1 + _viewport_y2 ) / 2;

   int viewport_radius_x = _viewport_cx - _viewport_x1;
   int viewport_radius_y = _viewport_cy - _viewport_y1;
   _viewport_radius_in_pixels = ( viewport_radius_x < viewport_radius_y )
      ? viewport_radius_x
      : viewport_radius_y;

   int viewport_width_in_pixels = _viewport_x2 - _viewport_x1 + 1;
   int viewport_height_in_pixels = _viewport_y2 - _viewport_y1 + 1;
   if ( viewport_width_in_pixels < viewport_height_in_pixels ) {
      _viewport_width = 2.0 * _viewport_radius;
      _viewport_height = _viewport_width
         * (viewport_height_in_pixels-1) / (float)(viewport_width_in_pixels-1);
   }
   else {
      _viewport_height = 2.0 * _viewport_radius;
      _viewport_width = _viewport_height
         * (viewport_width_in_pixels-1) / (float)(viewport_height_in_pixels-1);
   }

   // This function expects coordinates relative to an origin
   // in the lower left corner of the window (i.e. y+ is up).
   //
   glViewport(
      _viewport_x1,
      _window_height - _viewport_y2,
      _viewport_x2 - _viewport_x1,
      _viewport_y2 - _viewport_y1
   );
}

void Camera::zoomIn( float delta_pixels ) {

   // this should be a little greater than 1.0
   static const float magnificationFactorPerPixel = 1.005;

   _viewport_radius *= pow( magnificationFactorPerPixel, - delta_pixels );

   // recompute _viewport_width, _viewport_height
   //
   int viewport_width_in_pixels = _viewport_x2 - _viewport_x1 + 1;
   int viewport_height_in_pixels = _viewport_y2 - _viewport_y1 + 1;
   if ( viewport_width_in_pixels < viewport_height_in_pixels ) {
      _viewport_width = 2.0 * _viewport_radius;
      _viewport_height = _viewport_width
         * (viewport_height_in_pixels-1) / (float)(viewport_width_in_pixels-1);
   }
   else {
      _viewport_height = 2.0 * _viewport_radius;
      _viewport_width = _viewport_height
         * (viewport_width_in_pixels-1) / (float)(viewport_height_in_pixels-1);
   }
}

// ==================================================

// this should be larger than 1.0 (like maybe 2.0 or 3.0)
const float OrbitingCamera::_radiusMultiplier = 2.0;

// this should be a little greater than 0.0
const float OrbitingCamera::_nearPlaneFactor = 0.1;

// this should be a little greater than 1.0
const float OrbitingCamera::_fudgeFactor = 1.25;

OrbitingCamera::OrbitingCamera(
   int window_width, int window_height,
   int viewport_x1, int viewport_y1,
   int viewport_x2, int viewport_y2,
   float POI_x, float POI_y, float POI_z,
   float radius_of_scene
) :
   Camera(
      window_width, window_height,
      viewport_x1, viewport_y1, viewport_x2, viewport_y2,
      _fudgeFactor * _nearPlaneFactor * radius_of_scene / _radiusMultiplier,
      _nearPlaneFactor * radius_of_scene,
      2.0 * _radiusMultiplier * radius_of_scene
   ),
   _rotator( _viewport_cx, _viewport_cy, _viewport_radius_in_pixels ),
   _initial_distance_from_POI( _radiusMultiplier * radius_of_scene ),
   _initial_POI_x( POI_x ), _initial_POI_y( POI_y ), _initial_POI_z( POI_z ),
   _distance_from_POI( _initial_distance_from_POI ),
   _POI_x( _initial_POI_x ), _POI_y( _initial_POI_y ), _POI_z( _initial_POI_z )
{
}

void OrbitingCamera::reset() {

   Camera::reset();

   _distance_from_POI = _initial_distance_from_POI;
   _POI_x = _initial_POI_x;
   _POI_y = _initial_POI_y;
   _POI_z = _initial_POI_z;
}

void OrbitingCamera::invokeTransform() {

   Camera::invokeTransform();

   glTranslatef( 0.0, 0.0, - _distance_from_POI );

   glMultMatrixf( _rotator.getRotationMatrix() );

   // This is necessary for conversion between
   // z+ up (what the client expects) and
   // y+ up (what OpenGL expects)
   //
   glRotatef( - 90.0 , 1.0, 0.0, 0.0 );

   glTranslatef( - _POI_x, - _POI_y, - _POI_z );
}

void OrbitingCamera::resizeViewport(
   int window_width, int window_height,
   int viewport_x1, int viewport_y1,
   int viewport_x2, int viewport_y2
) {
   Camera::resizeViewport(
      window_width, window_height,
      viewport_x1, viewport_y1, viewport_x2, viewport_y2
   );

   _rotator.resize( _viewport_cx, _viewport_cy, _viewport_radius_in_pixels );
}

void OrbitingCamera::orbit(
   float old_x_pixels, float old_y_pixels,
   float new_x_pixels, float new_y_pixels
) {
   _rotator.deviceMotion(
      old_x_pixels, old_y_pixels, new_x_pixels, new_y_pixels
   );
}

void OrbitingCamera::translateSceneRightAndUp(
   float delta_x_pixels, float delta_y_pixels
) {
   float translationSpeedInUnitsPerRadius =
      _distance_from_POI * _viewport_radius / _near_plane;
   float pixelsPerUnit = _viewport_radius_in_pixels
      / translationSpeedInUnitsPerRadius;

   float inverseRotation[ 16 ], newTranslationMatrix[ 16 ];

   copyMatrix( inverseRotation, _rotator.getRotationMatrix() );
   transposeMatrix( inverseRotation );

   glMatrixMode( GL_MODELVIEW );
   glLoadIdentity();  // clear matrix
   glRotatef( 90.0 , 1.0, 0.0, 0.0 );
   glMultMatrixf( inverseRotation );
   glTranslatef(
      delta_x_pixels / pixelsPerUnit, delta_y_pixels / pixelsPerUnit, 0.0
   );
   glMultMatrixf( _rotator.getRotationMatrix() );
   glRotatef( - 90.0 , 1.0, 0.0, 0.0 );
   glTranslatef( - _POI_x, - _POI_y, - _POI_z );
   glGetFloatv( GL_MODELVIEW_MATRIX, newTranslationMatrix );

   /* The new translation matrix should look like
      1 0 0 x
      0 1 0 y
      0 0 1 z
      0 0 0 1
   */
   ASSERT_IS_EQUAL( newTranslationMatrix[  0 ], 1.0 );
   ASSERT_IS_EQUAL( newTranslationMatrix[  1 ], 0.0 );
   ASSERT_IS_EQUAL( newTranslationMatrix[  2 ], 0.0 );
   ASSERT_IS_EQUAL( newTranslationMatrix[  3 ], 0.0 );
   ASSERT_IS_EQUAL( newTranslationMatrix[  4 ], 0.0 );
   ASSERT_IS_EQUAL( newTranslationMatrix[  5 ], 1.0 );
   ASSERT_IS_EQUAL( newTranslationMatrix[  6 ], 0.0 );
   ASSERT_IS_EQUAL( newTranslationMatrix[  7 ], 0.0 );
   ASSERT_IS_EQUAL( newTranslationMatrix[  8 ], 0.0 );
   ASSERT_IS_EQUAL( newTranslationMatrix[  9 ], 0.0 );
   ASSERT_IS_EQUAL( newTranslationMatrix[ 10 ], 1.0 );
   ASSERT_IS_EQUAL( newTranslationMatrix[ 11 ], 0.0 );
   ASSERT_IS_EQUAL( newTranslationMatrix[ 15 ], 1.0 );

   _POI_x = - newTranslationMatrix[ 12 ];
   _POI_y = - newTranslationMatrix[ 13 ];
   _POI_z = - newTranslationMatrix[ 14 ];
}

void OrbitingCamera::dollyCameraForward( float delta_pixels ) {

   float translationSpeedInUnitsPerRadius =
      _distance_from_POI * _viewport_radius / _near_plane;
   float pixelsPerUnit = _viewport_radius_in_pixels
      / translationSpeedInUnitsPerRadius;

   _distance_from_POI -= delta_pixels / pixelsPerUnit;

   // prevent ugly clipping of the scene by the near camera plane
   //
   if ( _distance_from_POI < 1.001 * _near_plane )
      _distance_from_POI = 1.001 * _near_plane;
}

// ==================================================

FreeFloatingCamera::FreeFloatingCamera(
   int window_width, int window_height,
   int viewport_x1, int viewport_y1,
   int viewport_x2, int viewport_y2,
   float viewport_radius,
   float near_plane, float far_plane,
   float translationSpeedInUnitsPerRadius,
   float rotationSpeedInDegreesPerRadius
) :
   Camera(
      window_width, window_height,
      viewport_x1, viewport_y1, viewport_x2, viewport_y2,
      viewport_radius,
      near_plane, far_plane
   ),
   _translationSpeedInUnitsPerRadius( translationSpeedInUnitsPerRadius ),
   _rotationSpeedInDegreesPerRadius( rotationSpeedInDegreesPerRadius )
{
#ifdef SLOW
   assignToUnity( _cumulativeTransforms );
#else
   // This is necessary for conversion between
   // z+ up (what the client expects) and
   // y+ up (what OpenGL expects)
   //
   assignToRotation(
      _cumulativeTransforms,
      - 90.0 , 1.0, 0.0, 0.0
   );
#endif
}

void FreeFloatingCamera::reset() {

   Camera::reset();

#ifdef SLOW
   assignToUnity( _cumulativeTransforms );
#else
   assignToRotation(
      _cumulativeTransforms,
      - 90.0 , 1.0, 0.0, 0.0
   );
#endif
}

void FreeFloatingCamera::invokeTransform() {

   Camera::invokeTransform();

#ifdef SLOW
   // ...
   glRotatef( - 90.0 , 1.0, 0.0, 0.0 );
#endif
   glMultMatrixf( _cumulativeTransforms );
}

void FreeFloatingCamera::invokeTransformWithOffset(
   float position_x, float position_y, float position_z,
   float direction_x, float direction_y, float direction_z
) {
   Camera::invokeTransform();

#ifdef SLOW
   // ...
   glRotatef( - 90.0 , 1.0, 0.0, 0.0 );
#endif
   glMultMatrixf( _cumulativeTransforms );

   pushLevel(
      position_x, position_y, position_z,
      direction_x, direction_y, direction_z,
      true
   );
}

void FreeFloatingCamera::pitchCameraUp( float delta_pixels ) {

   float pixelsPerDegree = _viewport_radius_in_pixels
      / _rotationSpeedInDegreesPerRadius;

   premultiplyWithRotation(
      _cumulativeTransforms,
      - delta_pixels / pixelsPerDegree, 1.0, 0.0, 0.0
   );
}

void FreeFloatingCamera::yawCameraRight( float delta_pixels ) {

   float pixelsPerDegree = _viewport_radius_in_pixels
      / _rotationSpeedInDegreesPerRadius;

   premultiplyWithRotation(
      _cumulativeTransforms,
#ifdef SLOW
      delta_pixels / pixelsPerDegree, 0.0, 0.0, 1.0
#else
      delta_pixels / pixelsPerDegree, 0.0, 1.0, 0.0
#endif
   );
}

void FreeFloatingCamera::translateCameraRightAndUp(
   float delta_x_pixels, float delta_y_pixels
) {
   float pixelsPerUnit = _viewport_radius_in_pixels
      / _translationSpeedInUnitsPerRadius;

   premultiplyWithTranslation(
      _cumulativeTransforms,
#ifdef SLOW
      - delta_x_pixels / pixelsPerUnit, 0.0, - delta_y_pixels / pixelsPerUnit
#else
      - delta_x_pixels / pixelsPerUnit, - delta_y_pixels / pixelsPerUnit, 0.0
#endif
   );
}

void FreeFloatingCamera::dollyCameraForward( float delta_pixels ) {

   float pixelsPerUnit = _viewport_radius_in_pixels
      / _translationSpeedInUnitsPerRadius;

   premultiplyWithTranslation(
      _cumulativeTransforms,
#ifdef SLOW
      0.0, - delta_pixels / pixelsPerUnit, 0.0
#else
      0.0, 0.0, delta_pixels / pixelsPerUnit
#endif
   );
}

void FreeFloatingCamera::rollCameraRight( float delta_pixels ) {

   float pixelsPerDegree = _viewport_radius_in_pixels
      / _rotationSpeedInDegreesPerRadius;

   premultiplyWithRotation(
      _cumulativeTransforms,
#ifdef SLOW
      - delta_pixels / pixelsPerDegree, 0.0, 1.0, 0.0
#else
      delta_pixels / pixelsPerDegree, 0.0, 0.0, 1.0
#endif
   );
}

