
#include "randutil.h"
#include "global.h"

#include <sys/timeb.h> /* ftime() */

// --------------------------------------------------

void initRandom() { // uses the system date/time to initialize random seed
   struct timeb blah;
   unsigned int seed;
   ftime( &blah );
   /*
      blah.time is of type time_t, and is the number of seconds
      since the epoch (1970).  Assuming time_t is a 4-byte
      signed int, this won't wrap until 2038.

      blah.millitm is of type short, and provides us with
      millisecond resolution (i.e. it's between 0 and 999).

      We have to pass an *unsigned* int to srand().
      We could either pass blah.time, or (if we expect to
      be called multiple times per second), we could use the
      low-order bits of blah.time together with blah.millitm.

      In the former case, we could simply call
         srand( (unsigned int) time( NULL ) );
      However, we'll implement the latter case.
   */
   seed = (blah.time << 10) | blah.millitm;
   srand( seed );
}

// --------------------------------------------------

int randomInteger( int a, int b ) {

   ASSERT( b >= a );

   return a + (int)( (b-a+1.0) * ( rand() / (RAND_MAX + 1.0f) ) );
}

// --------------------------------------------------

float randomReal( float a, float b ) {

   return a + (b-a) * ( rand() / (float)RAND_MAX );
}

// --------------------------------------------------

void randomlyStir( int * array, int size ) {
   /*
      We wish to randomize (or randomly permute)
      the order of the elements within the given array.

      One reasonable strategy is to maintain two arrays
      or lists, a source and a destination.
      Initially, the source contains all the elements
      of the input array, and the destination is empty.
      We then pick random elements from the source and
      move them one-by-one to the end of the destination:

         for ( int i = 0; i < size; ++i ) {
            int j = ... // a random integer in [0,size-i-1]
            destination[i] = source[j];
            source.deleteElementAt( j );
         }

      Unfortunately, if we use a vector (i.e. an array) to store
      the source, the time required to delete an element is
      O(N), and if we use a linked list then the look-up time
      for an element is O(N).  Either way, the total running
      time is O(N^2).

      Observe, however, that since j is a uniformly chosen random
      number, is does not matter whether the ordering of
      elements in the source is conserved or not.  Hence,
      rather than deleting element j from the source,
      we could simply move the last element in the source
      to take the place of element j:

         for ( int i = 0; i < size; ++i ) {
            int j = ... // a random integer in [0,size-i-1]
            destination[i] = source[j];
            source[j] = source[ size-i-1 ];
         }

      Assuming that a vector is used to store the source, we
      now have a total running time of O(N).

      One further refinement is possible.  At each iteration of
      the loop, the source and destination contain (size-i) and (i)
      elements, respectively.  These sum to (size) elements,
      and hint that we may be able to perform the randomization
      *in place*, with only one array:

         for ( int i = 0; i < size; ++i ) {
            int j = ... // a random integer in [0,size-i-1]
            j += i; // now j is in [i,size-1]

            // swap elements i and j
            tmp = array[i];
            array[i] = array[j];
            array[j] = tmp;
         }

      Hence, after each iteration of the loop's body, the destination
      array is stored in elements [0,i], and the source array is stored
      in elements [i+1,size-1].

      Finally, as a small optimization, we see that the last iteration
      of the loop has no effect and thus may be skipped.
   */

   for ( int i = 0; i < size-1 /*skip last iteration*/; ++i ) {

      // a random integer in [0,size-i-1]
      int j = (int)( rand() / (RAND_MAX+1.0) * (size-i) );

      j += i; // now j is in [i,size-1]

      ASSERT( i <= j && j <= size-1 );

      // Swap elements i and j.
      // Note that i and j may be equal, which is okay
      // (we're not required to generate a derangement of the input).
      // Usually, however, i and j won't be equal,
      // hence we don't bother checking for it.
      int tmp = array[i];
      array[i] = array[j];
      array[j] = tmp;
   }
}

