
#include "Camera.h"
#include "drawutil.h"
#include "randutil.h"
#include "drawutil2D.h"

#include <GL/glut.h>

#include <list>
using namespace std;


float globalTime = 0; // a global pseudo-clock whose value is in [0,100]
Camera * camera = 0;
AlignedBox box( Point3(-1,-1,-1), Point3(1,1,1) );
bool isCameraAnimated = true;
bool isFrozen = false;


// colours
Point3 bgColour(0,0,0);
Point3 cubeColour(1,1,1);
Point3 strokeColour(1,0,0);
Point3 pointerColour(0,1,1);
Point3 letterColour(1,1,1);


// This contains an even number of points,
// representing line segments that have been drawn.
list< Point3 > points;
static const int MAX_NUM_LINE_SEGMENTS = 600; // must be an even number


bool LMB=false, MMB=false, RMB=false;
bool CTRL=false, ALT=false, SHIFT=false;
int mouse_x, mouse_y;
bool hasMouseBeenDragged = false;


#define MAX_LETTER_GRID_SIZE 5
class LetterGrid {
   struct Cell {
      float _age; // in [0,100]
      char _c;
      Cell() : _age( 0 ), _c( 0 ) { }
      Cell( float age, char c ) : _age( age ), _c( c ) { }
   };
   Cell _grid[ MAX_LETTER_GRID_SIZE ][ MAX_LETTER_GRID_SIZE ];
   int _size; // in [1, MAX_LETTER_GRID_SIZE]
   int _cursorX, _cursorY;
   Point3 _origin; // upper left corner of grid
   Vector3 _down, _right; // unit vectors
   float _edgeLength; // in world space units
public:
   LetterGrid(
      const Point3 & origin,
      const Vector3 & down, const Vector3 & right,
      float edgeLength
   ) :
      _size( 1 ),
      _cursorX( 0 ),
      _cursorY( 0 ),
      _origin( origin ),
      _down( down ),
      _right( right ),
      _edgeLength( edgeLength )
   {
      clearGrid();
   }
   void clearGrid() {
      for ( int x = 0; x < MAX_LETTER_GRID_SIZE; ++x )
         for ( int y = 0; y < MAX_LETTER_GRID_SIZE; ++y )
            _grid[x][y] = Cell();
   }
   Vector3 getFrontFacingVector() {
      return _down ^ _right;
   }
   void incrementAge() {
      for ( int x = 0; x < MAX_LETTER_GRID_SIZE; ++x )
         for ( int y = 0; y < MAX_LETTER_GRID_SIZE; ++y )
            if ( _grid[x][y]._c != 0 ) {
               _grid[x][y]._age += 0.3f;
               if ( _grid[x][y]._age > 100 )
                  _grid[x][y] = Cell(); // kill the cell
            }
   }
   void keyPress( char c ) {
      _grid[ _cursorX ][ _cursorY ] = Cell( 0, c );
      ++ _cursorX;
      if ( _cursorX >= _size ) {
         _cursorX = 0;
         ++ _cursorY;
         if ( _cursorY >= _size ) {
            _cursorY = 0;
            ++ _size;
            if ( _size > MAX_LETTER_GRID_SIZE ) {
               _size = 1;
            }
         }
      }
   }
   void draw( bool blinkCursor ) const {
      char buffer[2] = "x";
      for ( int x = 0; x < _size; ++x )
         for ( int y = 0; y < _size; ++y )
            if ( _grid[x][y]._c != 0 ) {
               glColor3fv( (
                  Vector3(letterColour)*(1-_grid[x][y]._age/100)
                  + Vector3(bgColour)*(_grid[x][y]._age/100)
               ).get() );
               buffer[0] = _grid[x][y]._c;
               drawString(
                  _origin + _right*(x*_edgeLength/_size)
                     + _down*((y+1)*_edgeLength/_size),
                  buffer,
                  _edgeLength/_size
                     * (1+0.5f*exp(-_grid[x][y]._age/20)
                              *cos(_grid[x][y]._age/5)),
                  _down ^ _right,
                  - _down,
                  true
               );
            }

      // draw the cursor

      //glDisable( GL_CULL_FACE );
      glDisable( GL_DEPTH_TEST );
      glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
      glEnable( GL_BLEND );
      glColor4f(
         letterColour.x(), letterColour.y(), letterColour.z(),
         blinkCursor
            ? 0.25f*(1+sin( 2*M_PI*globalTime/100 )) : 0.25f
      );
      float f = 0.2f;  // height of cursor, as a fraction of char height
      Point3 p0 = _origin + _right*(_cursorX*_edgeLength/_size)
         + _down*((_cursorY+1-f)*_edgeLength/_size);
      Point3 p1 = p0 + _right*(_edgeLength/_size);
      Point3 p2 = p1 + _down*(f*_edgeLength/_size);
      Point3 p3 = p0 + _down*(f*_edgeLength/_size);
      glBegin( GL_QUADS );
         glVertex3fv( p0.get() );
         glVertex3fv( p3.get() );
         glVertex3fv( p2.get() );
         glVertex3fv( p1.get() );
      glEnd();
      glDisable( GL_BLEND );
   }
};
vector< LetterGrid * > letterGrid;
int currentLetterGrid = 0; // the grid with keyboard focus


void computeCurrentLetterGrid() {
   Vector3 v = camera->getPosition() - Point3(0,0,0);
   float maximalDotProduct = -1000;
   currentLetterGrid = 0;
   for ( int i = 0; i < (int)letterGrid.size(); ++i ) {
      float candidateDotProduct = v * letterGrid[i]->getFrontFacingVector();
      if ( candidateDotProduct > maximalDotProduct ) {
         maximalDotProduct = candidateDotProduct;
         currentLetterGrid = i;
      }
   }
}


void reshapeCallback( int width, int height ) {
   if ( 0 != camera )
      camera->resizeViewport( width, height );
}

void drawCallback() {

   glClearColor( bgColour.x(), bgColour.y(), bgColour.z(), 0 );
   glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

   glDepthFunc( GL_LEQUAL );
   glEnable( GL_DEPTH_TEST );

   glEnable( GL_CULL_FACE );

   camera->transform();

   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();

   glDisable( GL_LIGHTING );

   // draw the cube

   glColor3fv( cubeColour.get() );
   glLineWidth( 5 );
   glPolygonMode( GL_FRONT_AND_BACK, GL_LINE );
   glutSolidCube( 2 );
   glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );

   // draw strokes

   glColor3fv( strokeColour.get() );
   list< Point3 >::const_iterator it = points.begin();
   glBegin( GL_LINES );
      for ( ; it != points.end(); ++it ) {
         glVertex3fv( (*it).get() );
         ++it;
         glVertex3fv( (*it).get() );
      }
   glEnd();

   // draw letter grids

   for ( int i = 0; i < (int)letterGrid.size(); ++i )
      letterGrid[i]->draw( i == currentLetterGrid );

   // draw the mouse pointer

   glDisable( GL_DEPTH_TEST );
   glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
   glEnable( GL_BLEND );
   glColor4f(
      pointerColour.x(), pointerColour.y(), pointerColour.z(),
      0.5f*(1+sin( 2*M_PI*globalTime/100 ))
   );
   {
      OpenGL2DInterface g;
      g.pushProjection(
         camera->getViewportWidthInPixels(),camera->getViewportHeightInPixels()
      );
      g.fillCircle( mouse_x, mouse_y, 10 );
      g.popProjection();
   }
   glDisable( GL_BLEND );

   // ----- finish up

   glutSwapBuffers();
}

void mouseCallback( int button, int state, int x, int y ) {
   int mod = glutGetModifiers();
   CTRL = mod & GLUT_ACTIVE_CTRL;
   ALT = mod & GLUT_ACTIVE_ALT;
   SHIFT = mod & GLUT_ACTIVE_SHIFT;

   if ( ! LMB && ! MMB && ! RMB && state == GLUT_DOWN ) {
      // this is the start of a click down, and may be the start of a drag
      hasMouseBeenDragged = false;
   }

   switch ( button ) {
      case GLUT_LEFT_BUTTON : LMB = state == GLUT_DOWN; break;
      case GLUT_MIDDLE_BUTTON : MMB = state == GLUT_DOWN; break;
      case GLUT_RIGHT_BUTTON : RMB = state == GLUT_DOWN; break;
   }

   if ( ! LMB && ! MMB && ! RMB && state == GLUT_UP ) {
      // this is the end of a release
      if ( ! hasMouseBeenDragged ) {
         // The mouse was not dragged at all.
         // Toggle camera animation.
         isCameraAnimated = ! isCameraAnimated;
      }
   }

   mouse_x = x;
   mouse_y = y;
}

void passiveMotionCallback( int x, int y ) {

   camera->orbit( mouse_x, mouse_y, x, y );

   computeCurrentLetterGrid();

   glutPostRedisplay();

   mouse_x = x;
   mouse_y = y;
   // printf("drag:(%d,%d)\n", mouse_x, mouse_y );
}

void motionCallback( int x, int y ) {
   hasMouseBeenDragged = true;

   Ray ray1 = camera->computeRay( mouse_x, mouse_y );
   mouse_x = x;
   mouse_y = y;
   Ray ray2 = camera->computeRay( mouse_x, mouse_y );

   Point3 point1, point2;

   // try intersecting rays with the box
   Vector3 v(0.02f,0.02f,0.02f);
   AlignedBox box2( box.getMin()-v, box.getMax()+v ); // slightly expand the box
   if (
      box2.intersectsExactly( ray1, point1 )
      && box2.intersectsExactly( ray2, point2 )
   ) {
      points.push_back( point1 );
      points.push_back( point2 );
      while ( (int)points.size() > MAX_NUM_LINE_SEGMENTS ) {
         // throw away the first line segment
         points.pop_front();
         points.pop_front();
      }
      glutPostRedisplay();
   }
}

void keyboardCallback( unsigned char key, int x, int y ) {
   if ( key == ' ' ) {
      // randomly reassign colours
      int i = randomInteger( 0, 7 );
      bgColour = Point3( 0.2f*(i&4), 0.2f*(i&2), 0.2f*(i&1) );
      i = randomInteger( 1, 7 );
      cubeColour = Point3( i&4, i&2, i&1 );
      i = randomInteger( 1, 7 );
      strokeColour = Point3( i&4, i&2, i&1 );
      i = randomInteger( 1, 7 );
      pointerColour = Point3( i&4, i&2, i&1 );
      i = randomInteger( 1, 7 );
      letterColour = Point3( i&4, i&2, i&1 );
      glutPostRedisplay();
      return;
   }
   if ( key < ' ' || key > '~' ) {
      key &= 127;
      if ( key <= ' ' ) key += ' '+1;
      else if ( key > '~' ) key = '~';
   }
   letterGrid[ currentLetterGrid ]->keyPress( key );
   glutPostRedisplay();
}

void specialCallback( int key, int x, int y ) {
#if 0 // for grabbing screenshots
   if ( key == GLUT_KEY_F1 ) {
      isFrozen = ! isFrozen;
      return;
   }
#endif
   keyboardCallback( key, x, y );
}

void idleCallback() {
   if ( ! isFrozen ) {
      globalTime += 0.5f;
      if ( globalTime > 100 ) globalTime = 0;

      if ( (int)globalTime % 5 == 0 && isCameraAnimated ) {
         int x = camera->getViewportWidthInPixels()/2;
         int y = camera->getViewportHeightInPixels()/2;
         camera->orbit( x, y, x+0.3f, y+0.3f );
         computeCurrentLetterGrid();
      }

      for ( int i = 0; i < (int)letterGrid.size(); ++i )
         letterGrid[i]->incrementAge();
      glutPostRedisplay();
   }
}

int main( int argc, char *argv[] ) {

   float d = 0.01f; // a small delta offset from the box's surface
   letterGrid.push_back( new LetterGrid(
      Point3(-1,1,1+d),Vector3(0,-1,0),Vector3(1,0,0), 2
   ) );
   letterGrid.push_back( new LetterGrid(
      Point3(1,1,-1-d),Vector3(0,-1,0),Vector3(-1,0,0), 2
   ) );
   letterGrid.push_back( new LetterGrid(
      Point3(1+d,1,1),Vector3(0,-1,0),Vector3(0,0,-1), 2
   ) );
   letterGrid.push_back( new LetterGrid(
      Point3(-1-d,1,-1),Vector3(0,-1,0),Vector3(0,0,1), 2
   ) );
   letterGrid.push_back( new LetterGrid(
      Point3(-1,1+d,-1),Vector3(0,0,1),Vector3(1,0,0), 2
   ) );
   letterGrid.push_back( new LetterGrid(
      Point3(-1,-1-d,1),Vector3(0,0,-1),Vector3(1,0,0), 2
   ) );

   glutInit( &argc, argv );
   glutInitDisplayMode( GLUT_RGB | GLUT_DEPTH | GLUT_DOUBLE );
   glutInitWindowSize( 512, 512 );
   glutCreateWindow("3D Toy for Toddlers");
   glutFullScreen();
   glutSetCursor( GLUT_CURSOR_NONE );

   glutDisplayFunc( drawCallback );
   glutReshapeFunc( reshapeCallback );
   glutKeyboardFunc( keyboardCallback );
   glutSpecialFunc( specialCallback );
   glutMouseFunc( mouseCallback );
   glutPassiveMotionFunc( passiveMotionCallback );
   glutMotionFunc( motionCallback );

   glutIdleFunc( idleCallback );

   int windowWidth = glutGet( GLUT_WINDOW_WIDTH );
   int windowHeight = glutGet( GLUT_WINDOW_HEIGHT );
   camera = new Camera(
      windowWidth, windowHeight, 1.6f, box.getCentre()
   );

   computeCurrentLetterGrid();

   glutMainLoop();
}

