
#include "drawutil2D.h"
#ifdef _WIN32
#include "global.h"  /* for M_PI */
#endif
#include <GL/glut.h>
#include <math.h>
#include <string.h>


/* #define USE_HACKISH_OFFSET */


// From the manual page for glutStrokeCharacter():
// The Mono Roman font supported by GLUT has characters that are
// 104.76 units wide, up to 119.05 units high, and descenders can
// go as low as 33.33 units.
static const float FONT_HEIGHT = 119.05f;
static const float CHAR_WIDTH = 104.76f;


void OpenGL2DInterface::resize( int w, int h ) {
   _window_width = w;
   _window_height = h;
   // Check out appendix H of the OpenGL Programming Guide for the
   // rationale behind this code.
   glViewport( 0, 0, w, h );
   glMatrixMode( GL_PROJECTION );
   glLoadIdentity();
   glOrtho( 0, w, 0, h, -1, 1 );
#ifdef USE_HACKISH_OFFSET
   glTranslatef( 0.375f, 0.375f, 0 );
#endif
}

void OpenGL2DInterface::pushProjection( int w, int h ) {
   _window_width = w;
   _window_height = h;
   glMatrixMode( GL_PROJECTION );
   glLoadIdentity();
   glOrtho( 0, w, 0, h, -1, 1 );
#ifdef USE_HACKISH_OFFSET
   glTranslatef( 0.375f, 0.375f, 0 );
#endif
   glMatrixMode( GL_MODELVIEW );
}

void OpenGL2DInterface::popProjection() {
   glMatrixMode( GL_PROJECTION );
   glPopMatrix();
   glMatrixMode( GL_MODELVIEW );
}

void OpenGL2DInterface::plotPixel( int x, int y ) const {
   y = _window_height - y - 1;
   glBegin( GL_POINTS );
#ifdef USE_HACKISH_OFFSET
      glVertex2i( x, y );
#else
      glVertex2f( x+0.5f, y+0.5f );
#endif
   glEnd();
}

void OpenGL2DInterface::drawLine( int x1, int y1, int x2, int y2 ) const {
   y1 = _window_height - y1 - 1;
   y2 = _window_height - y2 - 1;
   glBegin( GL_LINES );
#ifdef USE_HACKISH_OFFSET
      glVertex2i( x1, y1 );
      glVertex2i( x2, y2 );
#else
      glVertex2f( x1+0.5f, y1+0.5f );
      glVertex2f( x2+0.5f, y2+0.5f );
#endif
   glEnd();
   glBegin( GL_POINTS );
#ifdef USE_HACKISH_OFFSET
      glVertex2i( x2, y2 );
#else
      glVertex2f( x2+0.5f, y2+0.5f );
#endif
   glEnd();
}

void OpenGL2DInterface::drawRect( int x, int y, int w, int h ) const {
   y = _window_height - y - h;
   --w;
   --h;
   glBegin( GL_LINE_LOOP );
#ifdef USE_HACKISH_OFFSET
      glVertex2i( x, y );
      glVertex2i( x+w, y );
      glVertex2i( x+w, y+h );
      glVertex2i( x, y+h );
#else
      glVertex2f( x+0.5f, y+0.5f );
      glVertex2f( x+w+0.5f, y+0.5f );
      glVertex2f( x+w+0.5f, y+h+0.5f );
      glVertex2f( x+0.5f, y+h+0.5f );
#endif
   glEnd();
}

void OpenGL2DInterface::fillRect( int x, int y, int w, int h ) const {
   y = _window_height - y - h;
#if 0
   glBegin( GL_QUADS );
      glVertex2i( x, y );
      glVertex2i( x+w, y );
      glVertex2i( x+w, y+h );
      glVertex2i( x, y+h );
   glEnd();
#else
   glRecti( x, y, x+w, y+h );
#endif
}

void OpenGL2DInterface::drawCircle(
   int x, int y, int radius, bool filled
) const {
   y = _window_height - y - 1;
   if ( filled ) {
      glBegin( GL_TRIANGLE_FAN );
      glVertex2f( x+0.5f, y+0.5f );
   }
   else glBegin( GL_LINE_LOOP );
      int numSides = (int)( 2 * M_PI * radius + 1 );
      float deltaAngle = 2 * M_PI / numSides;

      // I used to loop up to "< numSides", but found that
      // the triangle fan did not produce a complete disc
      // (there was a 1-pixel-wide sliver missing).
      // After changing to "<=", it works fine.
      for ( int i = 0; i <= numSides; ++i ) {
         float angle = i * deltaAngle;
         glVertex2f( x+radius*cos(angle)+0.5f, y+radius*sin(angle)+0.5f );
      }
   glEnd();
}

void OpenGL2DInterface::fillCircle( int x, int y, int radius ) const {
   drawCircle( x, y, radius, true );
}

void OpenGL2DInterface::drawImage(
   int x1, int y1,
   const unsigned char * image,
   int width, int height, int numComponents
) const {

#if 1 /* use glDrawPixels() -- fast, but entire image is culled if raster pos is outside viewport.  A more flexible (but somewhat less efficient) alternative would be to render a textured quad. */

#if 0
   // Draws image upside down.  Furthermore, entire image is culled
   // if *lower* left corner is outside viewport.
   glRasterPos2i( x1, _window_height - height - y1 );
#else
   // Draws image right-side up.  However, entire image is culled
   // if *upper* left corner is outside viewport.
   glPixelZoom( 1.0f, -1.0f );
   glRasterPos2i( x1, _window_height - y1 );
#endif

   glPixelStorei( GL_PACK_ALIGNMENT, 1 );
   glPixelStorei( GL_UNPACK_ALIGNMENT, 1 );
   if ( numComponents == 1 )
      glDrawPixels( width, height, GL_LUMINANCE, GL_UNSIGNED_BYTE, image );
   else if ( numComponents == 3 )
      glDrawPixels( width, height, GL_RGB, GL_UNSIGNED_BYTE, image );
   else if ( numComponents == 4 )
      glDrawPixels( width, height, GL_RGBA, GL_UNSIGNED_BYTE, image );

#else /* plot individual GL_POINTS -- slow, but should always work */

   glBegin( GL_POINTS );
      if ( numComponents == 1 ) {
         unsigned char c;
         for ( int y = 0; y < height; ++y ) {
            for ( int x = 0; x < width; ++x ) {
               c = image[y*width+x];
               glColor3ub( c, c, c );
               glVertex2f( x1+x+0.5f, (_window_height - y-y1 - 1)+0.5f );
            }
         }
      }
      else if ( numComponents == 3 ) {
         for ( int y = 0; y < height; ++y ) {
            for ( int x = 0; x < width; ++x ) {
               glColor3ubv( &image[(y*width+x)*3 ] );
               glVertex2f( x1+x+0.5f, (_window_height - y-y1 - 1)+0.5f );
            }
         }
      }
   glEnd();

#endif
}

void OpenGL2DInterface::drawImage(
   int x1, int y1,
   const float * image,
   int width, int height, int numComponents,
   float min, float max
) const {
   // FIXME: this should be re-implemented using glDrawPixels().
   // To support the mapping from [min,max] to [0,1], use
   // glPixelTransfer()

   float delta = max-min;
   glBegin( GL_POINTS );
      if ( numComponents == 1 ) {
         for ( int y = 0; y < height; ++y ) {
            for ( int x = 0; x < width; ++x ) {
               float f = (image[y*width+x]-min)/delta;
               glColor3f( f, f, f );
               glVertex2f( x1+x+0.5f, (_window_height - y-y1 - 1)+0.5f );
            }
         }
      }
      else if ( numComponents == 3 ) {
         for ( int y = 0; y < height; ++y ) {
            for ( int x = 0; x < width; ++x ) {
               float f1 = (image[(y*width+x)*3  ]-min)/delta;
               float f2 = (image[(y*width+x)*3+1]-min)/delta;
               float f3 = (image[(y*width+x)*3+2]-min)/delta;
               glColor3f( f1, f2, f3 );
               glVertex2f( x1+x+0.5f, (_window_height - y-y1 - 1)+0.5f );
            }
         }
      }
   glEnd();
}

float OpenGL2DInterface::stringWidthInPixels(
   const char * buffer, int height
) {
   if ( buffer == 0 ) return 0;
   return height * strlen( buffer ) * CHAR_WIDTH / FONT_HEIGHT;
}

void OpenGL2DInterface::drawString(
   int x, int y,
   const char * buffer,
   int height,
   bool isBlended
) {
   if ( buffer == 0 ) return;

   y = _window_height - y - 1;

   glPushMatrix();
      glTranslatef( x+0.5f, y+0.5f, 0 );

      if ( isBlended ) {
         // This will draw the text with anti-aliased strokes,
         // making it easier to read small text.
         glEnable( GL_LINE_SMOOTH );
         glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
         glEnable( GL_BLEND );
      }

      // We scale the text to make its height that desired by the caller.
      float s = height / FONT_HEIGHT; // scale factor
      glScalef( s, s, 1 );
      for ( int j = 0; buffer[j] != '\0'; ++j )
         glutStrokeCharacter( GLUT_STROKE_MONO_ROMAN, buffer[j] );

      if ( isBlended ) {
         glDisable( GL_LINE_SMOOTH );
         glDisable( GL_BLEND );
      }
   glPopMatrix();
}

